"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.processAllRuleGaps = exports.PROCESS_GAPS_DEFAULT_PAGE_SIZE = void 0;
var _pMap = _interopRequireDefault(require("p-map"));
var _lodash = require("lodash");
var _find_gaps = require("./find_gaps");
var _constants = require("../../../common/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const PROCESS_GAPS_DEFAULT_PAGE_SIZE = exports.PROCESS_GAPS_DEFAULT_PAGE_SIZE = 500;
// Circuit breaker to prevent infinite loops
// It should be enough to update 5,000,000 gaps
// 10000 * 500 = 5,000,000 million gaps
const DEFAULT_MAX_ITERATIONS = 10000;
const getProcessingLimitsByRuleId = (countsByRuleId, maxProcessedGapsPerRule) => {
  const limits = {};
  if (!maxProcessedGapsPerRule) {
    return {};
  }
  Object.keys(countsByRuleId).forEach(ruleId => {
    limits[ruleId] = maxProcessedGapsPerRule - countsByRuleId[ruleId];
  });
  return limits;
};
const getNextRuleIdsToProcess = (overallProcessedCountsByRuleId, maxProcessedGapsPerRule) => {
  if (!maxProcessedGapsPerRule) {
    return Object.keys(overallProcessedCountsByRuleId);
  }
  return Object.entries(overallProcessedCountsByRuleId).filter(([_, count]) => count < maxProcessedGapsPerRule).map(([ruleId]) => ruleId);
};
const PROCESS_ALL_RULE_GAPS_CONCURRENCY = 10;
const PROCESS_ALL_RULE_GAPS_CHUNK_SIZE = 10;

/**
 * Fetches all gaps using search_after pagination to process more than 10,000 gaps with stable sorting
 */
const processAllRuleGaps = async ({
  ruleIds,
  start,
  end,
  statuses = [_constants.gapStatus.PARTIALLY_FILLED, _constants.gapStatus.UNFILLED],
  options,
  logger,
  eventLogClient,
  processGapsBatch
}) => {
  const processChunk = async ruleIdsToProcess => {
    let searchAfter;
    let pitId;
    let iterationCount = 0;
    const {
      maxProcessedGapsPerRule
    } = options !== null && options !== void 0 ? options : {};
    const overallProcessedCountsByRuleId = ruleIdsToProcess.reduce((acc, ruleId) => {
      acc[ruleId] = 0;
      return acc;
    }, {});
    try {
      while (true) {
        if (iterationCount >= DEFAULT_MAX_ITERATIONS) {
          logger.warn(`Circuit breaker triggered: Reached maximum number of iterations (${DEFAULT_MAX_ITERATIONS}) while processing gaps for rules ${ruleIdsToProcess.join(', ')}`);
          break;
        }
        iterationCount++;
        const gapsResponse = await (0, _find_gaps.findGapsSearchAfter)({
          eventLogClient,
          logger,
          params: {
            ruleIds: ruleIdsToProcess,
            start,
            end,
            perPage: PROCESS_GAPS_DEFAULT_PAGE_SIZE,
            statuses,
            sortField: '@timestamp',
            sortOrder: 'asc',
            searchAfter,
            pitId
          }
        });
        const {
          data: gapsToProcess,
          searchAfter: nextSearchAfter,
          pitId: nextPitId
        } = gapsResponse;
        pitId = nextPitId;
        if (gapsToProcess.length > 0) {
          const limitsByRuleId = getProcessingLimitsByRuleId(overallProcessedCountsByRuleId, maxProcessedGapsPerRule);
          const processedCountsByRuleId = await processGapsBatch(gapsToProcess, limitsByRuleId);
          Object.entries(processedCountsByRuleId).forEach(([ruleId, processedCount]) => {
            overallProcessedCountsByRuleId[ruleId] += processedCount;
          });
          ruleIdsToProcess = getNextRuleIdsToProcess(overallProcessedCountsByRuleId, maxProcessedGapsPerRule);
        }
        if (gapsToProcess.length === 0 || ruleIdsToProcess.length === 0 || !nextSearchAfter) {
          break;
        }
        searchAfter = nextSearchAfter;
      }
    } finally {
      if (pitId) {
        await eventLogClient.closePointInTime(pitId);
      }
    }
  };
  await (0, _pMap.default)((0, _lodash.chunk)(ruleIds, PROCESS_ALL_RULE_GAPS_CHUNK_SIZE), processChunk, {
    concurrency: PROCESS_ALL_RULE_GAPS_CONCURRENCY
  });
};
exports.processAllRuleGaps = processAllRuleGaps;