"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.RulesSettingsQueryDelayClient = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _boom = _interopRequireDefault(require("@hapi/boom"));
var _server = require("@kbn/core/server");
var _common = require("../../../common");
var _retry_if_conflicts = require("../../lib/retry_if_conflicts");
var _schemas = require("../schemas");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const verifyQueryDelaySettings = settings => {
  const {
    delay
  } = settings;
  if (delay < _common.MIN_QUERY_DELAY || delay > _common.MAX_QUERY_DELAY) {
    throw _boom.default.badRequest(`Invalid query delay value, must be between ${_common.MIN_QUERY_DELAY} and ${_common.MAX_QUERY_DELAY}, but got: ${delay}.`);
  }
};
class RulesSettingsQueryDelayClient {
  constructor(options) {
    (0, _defineProperty2.default)(this, "logger", void 0);
    (0, _defineProperty2.default)(this, "savedObjectsClient", void 0);
    (0, _defineProperty2.default)(this, "isServerless", void 0);
    (0, _defineProperty2.default)(this, "getModificationMetadata", void 0);
    this.logger = options.logger;
    this.savedObjectsClient = options.savedObjectsClient;
    this.isServerless = options.isServerless;
    this.getModificationMetadata = options.getModificationMetadata;
  }
  async get() {
    const rulesSettings = await this.getOrCreate();
    if (!rulesSettings.attributes.queryDelay) {
      this.logger.error('Failed to get query delay rules setting for current space.');
      throw new Error('Failed to get query delay rules setting for current space. Query delay settings are undefined');
    }
    return rulesSettings.attributes.queryDelay;
  }
  async update(newQueryDelayProperties) {
    return await (0, _retry_if_conflicts.retryIfConflicts)(this.logger, 'ruleSettingsClient.queryDelay.update()', async () => await this.updateWithOCC(newQueryDelayProperties));
  }
  async updateWithOCC(newQueryDelayProperties) {
    try {
      _schemas.queryDelaySchema.validate(newQueryDelayProperties);
      verifyQueryDelaySettings(newQueryDelayProperties);
    } catch (e) {
      this.logger.error(`Failed to verify new query delay settings properties when updating. Error: ${e}`);
      throw e;
    }
    const {
      attributes,
      version
    } = await this.getOrCreate();
    if (!attributes.queryDelay) {
      throw new Error('Query delay settings are undefined');
    }
    const modificationMetadata = await this.getModificationMetadata();
    try {
      const result = await this.savedObjectsClient.update(_common.RULES_SETTINGS_SAVED_OBJECT_TYPE, _common.RULES_SETTINGS_QUERY_DELAY_SAVED_OBJECT_ID, {
        queryDelay: {
          ...attributes.queryDelay,
          ...newQueryDelayProperties,
          updatedAt: modificationMetadata.updatedAt,
          updatedBy: modificationMetadata.updatedBy
        }
      }, {
        version
      });
      if (!result.attributes.queryDelay) {
        throw new Error('Query delay settings are undefined');
      }
      return result.attributes.queryDelay;
    } catch (e) {
      const errorMessage = 'savedObjectsClient errored trying to update query delay settings';
      this.logger.error(`${errorMessage}: ${e}`);
      throw _boom.default.boomify(e, {
        message: errorMessage
      });
    }
  }
  async getSettings() {
    return await this.savedObjectsClient.get(_common.RULES_SETTINGS_SAVED_OBJECT_TYPE, _common.RULES_SETTINGS_QUERY_DELAY_SAVED_OBJECT_ID);
  }
  async createSettings() {
    const modificationMetadata = await this.getModificationMetadata();
    const defaultQueryDelaySettings = this.isServerless ? _common.DEFAULT_SERVERLESS_QUERY_DELAY_SETTINGS : _common.DEFAULT_QUERY_DELAY_SETTINGS;
    try {
      return await this.savedObjectsClient.create(_common.RULES_SETTINGS_SAVED_OBJECT_TYPE, {
        queryDelay: {
          ...defaultQueryDelaySettings,
          ...modificationMetadata
        }
      }, {
        id: _common.RULES_SETTINGS_QUERY_DELAY_SAVED_OBJECT_ID,
        overwrite: true
      });
    } catch (e) {
      this.logger.error(`Failed to create query delay rules setting for current space. Error: ${e}`);
      throw e;
    }
  }

  /**
   * Helper function to ensure that a rules-settings saved object always exists.
   * Ensures the creation of the saved object is done lazily during retrieval.
   */
  async getOrCreate() {
    try {
      return await this.getSettings();
    } catch (e) {
      if (_server.SavedObjectsErrorHelpers.isNotFoundError(e)) {
        this.logger.info('Creating new default query delay rules settings for current space.');
        return await this.createSettings();
      }
      this.logger.error(`Failed to get query delay rules setting for current space. Error: ${e}`);
      throw e;
    }
  }
}
exports.RulesSettingsQueryDelayClient = RulesSettingsQueryDelayClient;