"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createGetHostServices = createGetHostServices;
var _range_query = require("@kbn/observability-utils-common/es/queries/range_query");
var _term_query = require("@kbn/observability-utils-common/es/queries/term_query");
var _es_fields = require("@kbn/apm-types/es_fields");
var _common = require("../../../common");
var _document_type = require("../../../common/document_type");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const MAX_SIZE = 1000;
const suitableTypes = [_document_type.ApmDocumentType.TransactionMetric, _document_type.ApmDocumentType.ErrorEvent];
function createGetHostServices({
  apmEventClient
}) {
  return async ({
    start,
    end,
    size = MAX_SIZE,
    filters,
    documentSources
  }) => {
    var _metricsQuery$aggrega, _logsQuery$aggregatio;
    const sourcesToUse = (0, _common.getPreferredBucketSizeAndDataSource)({
      sources: documentSources.filter(s => suitableTypes.includes(s.documentType)),
      bucketSizeInSeconds: (0, _common.getBucketSize)({
        start,
        end,
        numBuckets: 50
      }).bucketSize
    });
    const commonFiltersList = [...(0, _range_query.rangeQuery)(start, end), {
      exists: {
        field: _es_fields.SERVICE_NAME
      }
    }];
    if (filters[_es_fields.HOST_NAME]) {
      commonFiltersList.push({
        bool: {
          should: [...(0, _term_query.termQuery)(_es_fields.HOST_NAME, filters[_es_fields.HOST_NAME]), ...(0, _term_query.termQuery)(_es_fields.HOST_HOSTNAME, filters[_es_fields.HOST_HOSTNAME])],
          minimum_should_match: 1
        }
      });
    }
    // get services from transaction metrics
    const metricsQuery = await apmEventClient.search('get_apm_host_services_from_metrics', {
      apm: {
        sources: [{
          documentType: _document_type.ApmDocumentType.TransactionMetric,
          rollupInterval: _common.RollupInterval.OneMinute
        }]
      },
      track_total_hits: false,
      size: 0,
      query: {
        bool: {
          filter: [{
            bool: {
              should: [...(0, _term_query.termQuery)(_es_fields.METRICSET_NAME, 'app'), {
                bool: {
                  must: [...(0, _term_query.termQuery)(_es_fields.METRICSET_NAME, 'transaction')]
                }
              }],
              minimum_should_match: 1
            }
          }, ...commonFiltersList]
        }
      },
      aggs: {
        services: {
          terms: {
            field: _es_fields.SERVICE_NAME,
            size
          },
          aggs: {
            latestAgent: {
              top_metrics: {
                metrics: [{
                  field: _es_fields.AGENT_NAME
                }],
                sort: {
                  '@timestamp': 'desc'
                },
                size: 1
              }
            }
          }
        }
      }
    });

    // get services from logs
    const logsQuery = await apmEventClient.search('get_apm_host_services_from_logs', {
      apm: {
        sources: [{
          documentType: _document_type.ApmDocumentType.ErrorEvent,
          rollupInterval: sourcesToUse.source.rollupInterval
        }]
      },
      track_total_hits: false,
      size: 0,
      query: {
        bool: {
          filter: commonFiltersList
        }
      },
      aggs: {
        services: {
          terms: {
            field: _es_fields.SERVICE_NAME,
            size
          },
          aggs: {
            latestAgent: {
              top_metrics: {
                metrics: [{
                  field: _es_fields.AGENT_NAME
                }],
                sort: {
                  '@timestamp': 'desc'
                },
                size: 1
              }
            }
          }
        }
      }
    });
    const servicesListBucketsFromMetrics = ((_metricsQuery$aggrega = metricsQuery.aggregations) === null || _metricsQuery$aggrega === void 0 ? void 0 : _metricsQuery$aggrega.services.buckets) || [];
    const servicesListBucketsFromLogs = ((_logsQuery$aggregatio = logsQuery.aggregations) === null || _logsQuery$aggregatio === void 0 ? void 0 : _logsQuery$aggregatio.services.buckets) || [];
    const serviceMap = [...servicesListBucketsFromMetrics, ...servicesListBucketsFromLogs].reduce((acc, bucket) => {
      const serviceName = bucket.key;
      const latestAgentEntry = bucket.latestAgent.top[0];
      const latestTimestamp = latestAgentEntry.sort[0];
      const agentName = latestAgentEntry.metrics[_es_fields.AGENT_NAME];
      // dedup and get the latest timestamp
      const existingService = acc.get(serviceName);
      if (!existingService || existingService.latestTimestamp < latestTimestamp) {
        acc.set(serviceName, {
          latestTimestamp,
          agentName
        });
      }
      return acc;
    }, new Map());
    const services = Array.from(serviceMap).slice(0, size).map(([serviceName, {
      agentName
    }]) => ({
      serviceName,
      agentName
    }));
    return {
      services
    };
  };
}