"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.FORBIDDEN_SERVICE_NAMES = void 0;
exports.getServiceMapNodes = getServiceMapNodes;
var _lodash = require("lodash");
var _apm = require("../es_fields/apm");
var _group_resource_nodes = require("./group_resource_nodes");
var _utils = require("./utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const FORBIDDEN_SERVICE_NAMES = exports.FORBIDDEN_SERVICE_NAMES = ['constructor'];
function addMessagingConnections(connections, destinationServices) {
  const servicesByDestination = destinationServices.reduce((acc, {
    from,
    to
  }) => {
    var _acc$get;
    const key = from[_apm.SPAN_DESTINATION_SERVICE_RESOURCE];
    const currentDestinations = (_acc$get = acc.get(key)) !== null && _acc$get !== void 0 ? _acc$get : [];
    currentDestinations.push(to);
    acc.set(key, currentDestinations);
    return acc;
  }, new Map());
  const messagingConnections = connections.reduce((acc, connection) => {
    const destination = connection.destination;
    if ((0, _utils.isExitSpan)(destination) && destination[_apm.SPAN_TYPE] === 'messaging') {
      var _servicesByDestinatio;
      const matchedServices = (_servicesByDestinatio = servicesByDestination.get(destination[_apm.SPAN_DESTINATION_SERVICE_RESOURCE])) !== null && _servicesByDestinatio !== void 0 ? _servicesByDestinatio : [];
      matchedServices.forEach(matchedService => {
        acc.push({
          source: destination,
          destination: matchedService
        });
      });
    }
    return acc;
  }, []);
  return [...connections, ...messagingConnections];
}
function getAllNodes(services, connections) {
  const allNodesMap = new Map();
  connections.forEach(connection => {
    const sourceId = connection.source.id;
    const destinationId = connection.destination.id;
    if (!allNodesMap.has(sourceId)) {
      allNodesMap.set(sourceId, {
        ...connection.source,
        id: sourceId
      });
    }
    if (!allNodesMap.has(destinationId)) {
      allNodesMap.set(destinationId, {
        ...connection.destination,
        id: destinationId
      });
    }
  });

  // Derive the rest of the map nodes from the connections and add the services
  // from the services data query
  services.forEach(service => {
    const id = service[_apm.SERVICE_NAME];
    if (!allNodesMap.has(id) && !FORBIDDEN_SERVICE_NAMES.includes(service[_apm.SERVICE_NAME])) {
      allNodesMap.set(id, {
        ...service,
        id
      });
    }
  });
  return allNodesMap;
}
function getAllServices(allNodes, destinationServices, anomalies) {
  const anomaliesByServiceName = new Map(anomalies.serviceAnomalies.map(item => [item.serviceName, item]));
  const serviceNodes = new Map();
  for (const {
    from,
    to
  } of destinationServices) {
    const fromId = from.id;
    const toId = to.id;
    if (allNodes.has(fromId) && !allNodes.has(toId)) {
      serviceNodes.set(toId, {
        ...to,
        id: toId,
        serviceAnomalyStats: anomaliesByServiceName.get(to.id)
      });
    }
  }
  for (const node of allNodes.values()) {
    if (!(0, _utils.isExitSpan)(node)) {
      serviceNodes.set(node.id, {
        ...node,
        serviceAnomalyStats: anomaliesByServiceName.get(node.id)
      });
    }
  }
  return serviceNodes;
}
function getExitSpans(allNodes) {
  const exitSpans = new Map();
  for (const node of allNodes.values()) {
    if ((0, _utils.isExitSpan)(node)) {
      var _exitSpans$get;
      const nodes = (_exitSpans$get = exitSpans.get(node.id)) !== null && _exitSpans$get !== void 0 ? _exitSpans$get : [];
      nodes.push(node);
      exitSpans.set(node.id, nodes);
    }
  }
  return exitSpans;
}
function exitSpanDestinationsToMap(destinationServices) {
  return destinationServices.reduce((acc, {
    from,
    to
  }) => {
    acc.set(from.id, to);
    return acc;
  }, new Map());
}
function mapNodes({
  allConnections,
  nodes,
  exitSpanDestinations,
  services
}) {
  const exitSpanDestinationsMap = exitSpanDestinationsToMap(exitSpanDestinations);
  const exitSpans = getExitSpans(nodes);
  const messagingSpanIds = new Set(allConnections.filter(({
    source
  }) => (0, _utils.isExitSpan)(source)).map(({
    source
  }) => source.id));

  // 1. Map external nodes to internal services
  // 2. Collapse external nodes into one node based on span.destination.service.resource
  // 3. Pick the first available span.type/span.subtype in an alphabetically sorted list
  const mappedNodes = new Map();
  for (const [id, node] of nodes.entries()) {
    if (mappedNodes.has(id)) {
      continue;
    }
    const isMessagingSpan = messagingSpanIds.has(node.id);
    const destinationService = isMessagingSpan ? undefined : exitSpanDestinationsMap.get(node.id);
    const isServiceNode = !!destinationService || !(0, _utils.isExitSpan)(node);
    if (isServiceNode) {
      const serviceId = destinationService ? destinationService.id : node.id;
      const serviceNode = services.get(serviceId);
      if (serviceNode) {
        mappedNodes.set(node.id, serviceNode);
      }
    } else {
      var _exitSpans$get2;
      const exitSpanNodes = (_exitSpans$get2 = exitSpans.get(id)) !== null && _exitSpans$get2 !== void 0 ? _exitSpans$get2 : [];
      if (exitSpanNodes.length > 0) {
        const exitSpanSample = exitSpanNodes[0];
        mappedNodes.set(id, {
          ...exitSpanSample,
          id: (0, _utils.getExitSpanNodeId)(exitSpanSample),
          label: exitSpanSample[_apm.SPAN_DESTINATION_SERVICE_RESOURCE],
          [_apm.SPAN_TYPE]: exitSpanNodes.map(n => n[_apm.SPAN_TYPE]).sort()[0],
          [_apm.SPAN_SUBTYPE]: exitSpanNodes.map(n => n[_apm.SPAN_SUBTYPE]).sort()[0]
        });
      }
    }
  }
  return mappedNodes;
}
function mapEdges({
  allConnections,
  nodes
}) {
  const connections = allConnections.reduce((acc, connection) => {
    const sourceData = nodes.get(connection.source.id);
    const targetData = nodes.get(connection.destination.id);
    if (!sourceData || !targetData || sourceData.id === targetData.id) {
      return acc;
    }
    const label = `${sourceData[_apm.SERVICE_NAME] || sourceData[_apm.SPAN_DESTINATION_SERVICE_RESOURCE]} to ${targetData[_apm.SERVICE_NAME] || targetData[_apm.SPAN_DESTINATION_SERVICE_RESOURCE]}`;
    const id = (0, _utils.getEdgeId)(sourceData.id, targetData.id);
    acc.set(id, {
      source: sourceData.id,
      target: targetData.id,
      label,
      id,
      sourceData,
      targetData
    });
    return acc;
  }, new Map());
  return [...connections.values()];
}
function markBidirectionalConnections({
  connections
}) {
  const targets = new Map();
  for (const connection of connections) {
    const edgeId = (0, _utils.getEdgeId)(connection.source, connection.target);
    const reversedConnection = targets.get(edgeId);
    if (reversedConnection) {
      reversedConnection.bidirectional = true;
      connection.isInverseEdge = true;
    }
    targets.set(edgeId, connection);
  }
  return targets.values();
}
function getServiceMapNodes({
  connections,
  exitSpanDestinations,
  servicesData,
  anomalies
}) {
  const allConnections = addMessagingConnections(connections, exitSpanDestinations);
  const allNodes = getAllNodes(servicesData, allConnections);
  const allServices = getAllServices(allNodes, exitSpanDestinations, anomalies);
  const nodes = mapNodes({
    allConnections,
    nodes: allNodes,
    services: allServices,
    exitSpanDestinations
  });

  // Build connections with mapped nodes
  const mappedEdges = mapEdges({
    allConnections,
    nodes
  });
  const uniqueNodes = mappedEdges.flatMap(connection => [connection.sourceData, connection.targetData]).concat(...allServices.values()).reduce((acc, node) => {
    if (!acc.has(node.id)) {
      acc.set(node.id, node);
    }
    return acc;
  }, new Map()).values();

  // Instead of adding connections in two directions,
  // we add a `bidirectional` flag to use in styling
  const edges = markBidirectionalConnections({
    connections: (0, _lodash.sortBy)(mappedEdges, 'id')
  });

  // Put everything together in elements, with everything in the "data" property
  const elements = [...edges, ...uniqueNodes].map(element => ({
    data: element
  }));
  return (0, _group_resource_nodes.groupResourceNodes)({
    elements
  });
}