"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useFailedTransactionsCorrelations = useFailedTransactionsCorrelations;
var _react = require("react");
var _lodash = require("lodash");
var _apm = require("../../../../common/es_fields/apm");
var _event_outcome = require("../../../../common/event_outcome");
var _constants = require("../../../../common/correlations/constants");
var _latency_distribution_chart_types = require("../../../../common/latency_distribution_chart_types");
var _create_call_apm_api = require("../../../services/rest/create_call_apm_api");
var _analysis_hook_utils = require("./utils/analysis_hook_utils");
var _use_fetch_params = require("./use_fetch_params");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

// Overall progress is a float from 0 to 1.
const LOADED_OVERALL_HISTOGRAM = 0.05;
const LOADED_ERROR_HISTOGRAM = LOADED_OVERALL_HISTOGRAM + 0.05;
const LOADED_FIELD_CANDIDATES = LOADED_ERROR_HISTOGRAM + 0.05;
const LOADED_DONE = 1;
const PROGRESS_STEP_P_VALUES = 0.9 - LOADED_FIELD_CANDIDATES;
function useFailedTransactionsCorrelations() {
  const fetchParams = (0, _use_fetch_params.useFetchParams)();

  // This use of useReducer (the dispatch function won't get reinstantiated
  // on every update) and debounce avoids flooding consuming components with updates.
  // `setResponse.flush()` can be used to enforce an update.
  const [response, setResponseUnDebounced] = (0, _react.useReducer)((0, _analysis_hook_utils.getReducer)(), (0, _analysis_hook_utils.getInitialResponse)());
  const setResponse = (0, _react.useMemo)(() => (0, _lodash.debounce)(setResponseUnDebounced, _constants.DEBOUNCE_INTERVAL), []);
  const abortCtrl = (0, _react.useRef)(new AbortController());
  const startFetch = (0, _react.useCallback)(async () => {
    abortCtrl.current.abort();
    abortCtrl.current = new AbortController();
    setResponse({
      ...(0, _analysis_hook_utils.getInitialResponse)(),
      isRunning: true,
      // explicitly set these to undefined to override a possible previous state.
      error: undefined,
      failedTransactionsCorrelations: undefined,
      percentileThresholdValue: undefined,
      overallHistogram: undefined,
      totalDocCount: undefined,
      errorHistogram: undefined
    });
    setResponse.flush();
    try {
      // `responseUpdate` will be enriched with additional data with subsequent
      // calls to the overall histogram, field candidates, field value pairs, correlation results
      // and histogram data for statistically significant results.
      const responseUpdate = {
        ccsWarning: false,
        fallbackResult: undefined
      };

      // Initial call to fetch the overall distribution for the log-log plot.
      const overallHistogramResponse = await (0, _create_call_apm_api.callApmApi)('POST /internal/apm/latency/overall_distribution/transactions', {
        signal: abortCtrl.current.signal,
        params: {
          body: {
            ...fetchParams,
            percentileThreshold: _constants.DEFAULT_PERCENTILE_THRESHOLD,
            chartType: _latency_distribution_chart_types.LatencyDistributionChartType.failedTransactionsCorrelations
          }
        }
      });
      if (abortCtrl.current.signal.aborted) {
        return;
      }
      const {
        overallHistogram,
        totalDocCount,
        percentileThresholdValue,
        durationMin,
        durationMax
      } = overallHistogramResponse;
      responseUpdate.overallHistogram = overallHistogram;
      responseUpdate.totalDocCount = totalDocCount;
      responseUpdate.percentileThresholdValue = percentileThresholdValue;
      setResponse({
        ...responseUpdate,
        loaded: LOADED_OVERALL_HISTOGRAM
      });
      setResponse.flush();
      const errorHistogramResponse = await (0, _create_call_apm_api.callApmApi)('POST /internal/apm/latency/overall_distribution/transactions', {
        signal: abortCtrl.current.signal,
        params: {
          body: {
            ...fetchParams,
            percentileThreshold: _constants.DEFAULT_PERCENTILE_THRESHOLD,
            termFilters: [{
              fieldName: _apm.EVENT_OUTCOME,
              fieldValue: _event_outcome.EventOutcome.failure
            }],
            durationMin,
            durationMax,
            chartType: _latency_distribution_chart_types.LatencyDistributionChartType.failedTransactionsCorrelations
          }
        }
      });
      if (abortCtrl.current.signal.aborted) {
        return;
      }
      const {
        overallHistogram: errorHistogram
      } = errorHistogramResponse;
      responseUpdate.errorHistogram = errorHistogram;
      setResponse({
        ...responseUpdate,
        loaded: LOADED_ERROR_HISTOGRAM
      });
      setResponse.flush();
      const {
        fieldCandidates: candidates
      } = await (0, _create_call_apm_api.callApmApi)('GET /internal/apm/correlations/field_candidates/transactions', {
        signal: abortCtrl.current.signal,
        params: {
          query: fetchParams
        }
      });
      if (abortCtrl.current.signal.aborted) {
        return;
      }
      const fieldCandidates = candidates.filter(t => !(t === _apm.EVENT_OUTCOME));
      setResponse({
        loaded: LOADED_FIELD_CANDIDATES
      });
      setResponse.flush();
      const failedTransactionsCorrelations = [];
      let fallbackResult;
      const fieldsToSample = new Set();
      const chunkSize = 10;
      let chunkLoadCounter = 0;
      const fieldCandidatesChunks = (0, _lodash.chunk)(fieldCandidates, chunkSize);
      for (const fieldCandidatesChunk of fieldCandidatesChunks) {
        const pValues = await (0, _create_call_apm_api.callApmApi)('POST /internal/apm/correlations/p_values/transactions', {
          signal: abortCtrl.current.signal,
          params: {
            body: {
              ...fetchParams,
              fieldCandidates: fieldCandidatesChunk,
              durationMin,
              durationMax
            }
          }
        });
        if (pValues.failedTransactionsCorrelations.length > 0) {
          pValues.failedTransactionsCorrelations.forEach(d => {
            fieldsToSample.add(d.fieldName);
          });
          failedTransactionsCorrelations.push(...pValues.failedTransactionsCorrelations);
          responseUpdate.failedTransactionsCorrelations = (0, _analysis_hook_utils.getFailedTransactionsCorrelationsSortedByScore)([...failedTransactionsCorrelations]);
        } else {
          // If there's no significant correlations found and there's a fallback result
          // Update the highest ranked/scored fall back result
          if (pValues.fallbackResult) {
            if (!fallbackResult) {
              fallbackResult = pValues.fallbackResult;
            } else {
              if (pValues.fallbackResult.normalizedScore > fallbackResult.normalizedScore) {
                fallbackResult = pValues.fallbackResult;
              }
            }
          }
        }
        chunkLoadCounter++;
        setResponse({
          ...responseUpdate,
          loaded: LOADED_FIELD_CANDIDATES + chunkLoadCounter / fieldCandidatesChunks.length * PROGRESS_STEP_P_VALUES
        });
        if (abortCtrl.current.signal.aborted) {
          return;
        }
      }
      setResponse({
        ...responseUpdate,
        fallbackResult,
        loaded: LOADED_DONE,
        isRunning: false
      });
      setResponse.flush();
    } catch (e) {
      if (!abortCtrl.current.signal.aborted) {
        var _err$body$message, _err$body, _err$response;
        const err = e;
        setResponse({
          error: 'response' in err ? (_err$body$message = (_err$body = err.body) === null || _err$body === void 0 ? void 0 : _err$body.message) !== null && _err$body$message !== void 0 ? _err$body$message : (_err$response = err.response) === null || _err$response === void 0 ? void 0 : _err$response.statusText : err.message,
          isRunning: false
        });
        setResponse.flush();
      }
    }
  }, [fetchParams, setResponse]);
  const cancelFetch = (0, _react.useCallback)(() => {
    abortCtrl.current.abort();
    setResponse({
      isRunning: false
    });
    setResponse.flush();
  }, [setResponse]);

  // auto-update
  (0, _react.useEffect)(() => {
    startFetch();
    return () => {
      abortCtrl.current.abort();
    };
  }, [startFetch, cancelFetch]);
  const {
    error,
    loaded,
    isRunning,
    ...returnedResponse
  } = response;
  const progress = (0, _react.useMemo)(() => ({
    error,
    loaded: Math.round(loaded * 100) / 100,
    isRunning
  }), [error, loaded, isRunning]);
  return {
    progress,
    response: returnedResponse,
    startFetch,
    cancelFetch
  };
}