"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getDeprecationsInfo = getDeprecationsInfo;
var _i18n = require("@kbn/i18n");
var _deprecations = require("../lib/deprecations");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const APM_USER_ROLE_NAME = 'apm_user';
const getKibanaPrivilegesDocumentationUrl = branch => {
  return `https://www.elastic.co/guide/en/kibana/${branch}/kibana-privileges.html`;
};
async function getDeprecationsInfo({
  esClient
}, core, apmDeps) {
  const client = esClient.asCurrentUser;
  const {
    docLinks
  } = core;
  const {
    security
  } = apmDeps;

  // Nothing to do if security is disabled
  if (!(security !== null && security !== void 0 && security.license.isEnabled())) {
    return [];
  }
  const [userDeprecations, roleMappingDeprecations] = await Promise.all([getUsersDeprecations(client, apmDeps, docLinks), getRoleMappingsDeprecations(client, apmDeps, docLinks)]);
  return [...userDeprecations, ...roleMappingDeprecations];
}
async function getUsersDeprecations(client, apmDeps, docLinks) {
  const title = _i18n.i18n.translate('xpack.apm.deprecations.apmUser.title', {
    defaultMessage: `Check for users assigned the deprecated "{apmUserRoleName}" role`,
    values: {
      apmUserRoleName: APM_USER_ROLE_NAME
    }
  });
  let users;
  try {
    users = await client.security.getUser();
  } catch (err) {
    const {
      logger
    } = apmDeps;
    if (_deprecations.deprecations.getErrorStatusCode(err) === 403) {
      logger.warn('Failed to retrieve users when checking for deprecations: the "read_security" or "manage_security" cluster privilege is required.');
    } else {
      logger.error(`Failed to retrieve users when checking for deprecations, unexpected error: ${_deprecations.deprecations.getDetailedErrorMessage(err)}.`);
    }
    return _deprecations.deprecations.deprecationError(title, err, docLinks);
  }
  const apmUsers = Object.values(users).flatMap(user => user.roles.find(hasApmUserRole) ? user.username : []);
  if (apmUsers.length === 0) {
    return [];
  }
  return [{
    title,
    message: _i18n.i18n.translate('xpack.apm.deprecations.apmUser.description', {
      defaultMessage: `The "{apmUserRoleName}" role has been deprecated. Remove the "{apmUserRoleName}" role from affected users in this cluster including: {users}`,
      values: {
        apmUserRoleName: APM_USER_ROLE_NAME,
        users: apmUsers.join()
      }
    }),
    correctiveActions: {
      manualSteps: [_i18n.i18n.translate('xpack.apm.deprecations.apmUser.manualStepOne', {
        defaultMessage: `Go to Management > Security > Users to find users with the "{apmUserRoleName}" role.`,
        values: {
          apmUserRoleName: APM_USER_ROLE_NAME
        }
      }), _i18n.i18n.translate('xpack.apm.deprecations.apmUser.manualStepTwo', {
        defaultMessage: 'Remove the "{apmUserRoleName}" role from all users and add the built-in "viewer" role.',
        values: {
          apmUserRoleName: APM_USER_ROLE_NAME
        }
      })]
    },
    level: 'critical',
    deprecationType: 'feature',
    documentationUrl: getKibanaPrivilegesDocumentationUrl(docLinks.version)
  }];
}
async function getRoleMappingsDeprecations(client, apmDeps, docLinks) {
  const title = _i18n.i18n.translate('xpack.apm.deprecations.apmUserRoleMappings.title', {
    defaultMessage: `Check for role mappings using the deprecated "{apmUserRoleName}" role`,
    values: {
      apmUserRoleName: APM_USER_ROLE_NAME
    }
  });
  let roleMappings;
  try {
    roleMappings = await client.security.getRoleMapping();
  } catch (err) {
    const {
      logger
    } = apmDeps;
    if (_deprecations.deprecations.getErrorStatusCode(err) === 403) {
      logger.warn('Failed to retrieve role mappings when checking for deprecations: the "manage_security" cluster privilege is required.');
    } else {
      logger.error(`Failed to retrieve role mappings when checking for deprecations, unexpected error: ${_deprecations.deprecations.getDetailedErrorMessage(err)}.`);
    }
    return _deprecations.deprecations.deprecationError(title, err, docLinks);
  }
  const roleMappingsWithApmUserRole = Object.entries(roleMappings).flatMap(([roleName, role]) => {
    var _role$roles;
    return (_role$roles = role.roles) !== null && _role$roles !== void 0 && _role$roles.find(hasApmUserRole) ? roleName : [];
  });
  if (roleMappingsWithApmUserRole.length === 0) {
    return [];
  }
  return [{
    title,
    message: _i18n.i18n.translate('xpack.apm.deprecations.apmUserRoleMappings.description', {
      defaultMessage: `The "{apmUserRoleName}" role has been deprecated. Remove the "{apmUserRoleName}" role from affected role mappings in this cluster including: {roles}`,
      values: {
        apmUserRoleName: APM_USER_ROLE_NAME,
        roles: roleMappingsWithApmUserRole.join()
      }
    }),
    correctiveActions: {
      manualSteps: [_i18n.i18n.translate('xpack.apm.deprecations.apmUserRoleMappings.manualStepOne', {
        defaultMessage: `Go to Management > Security > Role Mappings to find roles mappings with the "{apmUserRoleName}" role.`,
        values: {
          apmUserRoleName: APM_USER_ROLE_NAME
        }
      }), _i18n.i18n.translate('xpack.apm.deprecations.apmUserRoleMappings.manualStepTwo', {
        defaultMessage: 'Remove the "{apmUserRoleName}" role from all role mappings and add the built-in "viewer" role',
        values: {
          apmUserRoleName: APM_USER_ROLE_NAME
        }
      })]
    },
    level: 'critical',
    deprecationType: 'feature',
    documentationUrl: getKibanaPrivilegesDocumentationUrl(docLinks.version)
  }];
}
const hasApmUserRole = role => role === APM_USER_ROLE_NAME;