"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getMobileDetailedStatisticsByFieldPeriods = getMobileDetailedStatisticsByFieldPeriods;
var _server = require("@kbn/observability-plugin/server");
var _lodash = require("lodash");
var _get_bucket_size = require("../../../common/utils/get_bucket_size");
var _get_offset_in_ms = require("../../../common/utils/get_offset_in_ms");
var _environment_query = require("../../../common/utils/environment_query");
var _apm = require("../../../common/es_fields/apm");
var _latency_aggregation_type = require("../../lib/helpers/latency_aggregation_type");
var _latency_aggregation_types = require("../../../common/latency_aggregation_types");
var _offset_previous_period_coordinate = require("../../../common/utils/offset_previous_period_coordinate");
var _document_type = require("../../../common/document_type");
var _rollup = require("../../../common/rollup");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function getMobileDetailedStatisticsByField({
  environment,
  kuery,
  serviceName,
  field,
  fieldValues,
  apmEventClient,
  start,
  end,
  offset
}) {
  var _response$aggregation, _response$aggregation2;
  const {
    startWithOffset,
    endWithOffset
  } = (0, _get_offset_in_ms.getOffsetInMs)({
    start,
    end,
    offset
  });
  const {
    intervalString
  } = (0, _get_bucket_size.getBucketSize)({
    start: startWithOffset,
    end: endWithOffset,
    minBucketSize: 60
  });
  const response = await apmEventClient.search(`get_mobile_detailed_statistics_by_field`, {
    apm: {
      sources: [{
        documentType: _document_type.ApmDocumentType.TransactionEvent,
        rollupInterval: _rollup.RollupInterval.None
      }]
    },
    track_total_hits: false,
    size: 0,
    query: {
      bool: {
        filter: [...(0, _server.termQuery)(_apm.SERVICE_NAME, serviceName), ...(0, _server.rangeQuery)(startWithOffset, endWithOffset), ...(0, _environment_query.environmentQuery)(environment), ...(0, _server.kqlQuery)(kuery)]
      }
    },
    aggs: {
      detailed_statistics: {
        terms: {
          field,
          include: fieldValues,
          size: fieldValues.length
        },
        aggs: {
          timeseries: {
            date_histogram: {
              field: '@timestamp',
              fixed_interval: intervalString,
              min_doc_count: 0,
              extended_bounds: {
                min: startWithOffset,
                max: endWithOffset
              }
            },
            aggs: {
              latency: {
                avg: {
                  field: _apm.TRANSACTION_DURATION
                }
              }
            }
          }
        }
      }
    }
  });
  const buckets = (_response$aggregation = (_response$aggregation2 = response.aggregations) === null || _response$aggregation2 === void 0 ? void 0 : _response$aggregation2.detailed_statistics.buckets) !== null && _response$aggregation !== void 0 ? _response$aggregation : [];
  return buckets.map(bucket => {
    const fieldName = bucket.key;
    const latency = bucket.timeseries.buckets.map(timeseriesBucket => ({
      x: timeseriesBucket.key,
      y: (0, _latency_aggregation_type.getLatencyValue)({
        latencyAggregationType: _latency_aggregation_types.LatencyAggregationType.avg,
        aggregation: timeseriesBucket.latency
      })
    }));
    const throughput = bucket.timeseries.buckets.map(timeseriesBucket => ({
      x: timeseriesBucket.key,
      y: timeseriesBucket.doc_count
    }));
    return {
      fieldName,
      latency,
      throughput
    };
  });
}
async function getMobileDetailedStatisticsByFieldPeriods({
  environment,
  kuery,
  serviceName,
  field,
  fieldValues,
  apmEventClient,
  start,
  end,
  offset
}) {
  const commonProps = {
    environment,
    kuery,
    serviceName,
    field,
    fieldValues,
    apmEventClient,
    start,
    end
  };
  const currentPeriodPromise = getMobileDetailedStatisticsByField({
    ...commonProps
  });
  const previousPeriodPromise = offset ? getMobileDetailedStatisticsByField({
    ...commonProps,
    offset
  }) : [];
  const [currentPeriod, previousPeriod] = await Promise.all([currentPeriodPromise, previousPeriodPromise]);
  const firstCurrentPeriod = currentPeriod === null || currentPeriod === void 0 ? void 0 : currentPeriod[0];
  return {
    currentPeriod: (0, _lodash.keyBy)(currentPeriod, 'fieldName'),
    previousPeriod: (0, _lodash.keyBy)(previousPeriod.map(data => {
      return {
        ...data,
        latency: (0, _offset_previous_period_coordinate.offsetPreviousPeriodCoordinates)({
          currentPeriodTimeseries: firstCurrentPeriod === null || firstCurrentPeriod === void 0 ? void 0 : firstCurrentPeriod.latency,
          previousPeriodTimeseries: data.latency
        }),
        throughput: (0, _offset_previous_period_coordinate.offsetPreviousPeriodCoordinates)({
          currentPeriodTimeseries: firstCurrentPeriod === null || firstCurrentPeriod === void 0 ? void 0 : firstCurrentPeriod.throughput,
          previousPeriodTimeseries: data.throughput
        })
      };
    }), 'fieldName')
  };
}