"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getServiceAgent = getServiceAgent;
var _server = require("@kbn/observability-plugin/server");
var _common = require("@kbn/observability-plugin/common");
var _utils = require("@kbn/apm-data-access-plugin/server/utils");
var _as_mutable_array = require("../../../common/utils/as_mutable_array");
var _apm = require("../../../common/es_fields/apm");
var _serverless = require("../../../common/serverless");
var _maybe = require("../../../common/utils/maybe");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function getServiceAgent({
  serviceName,
  apmEventClient,
  start,
  end
}) {
  const fields = (0, _as_mutable_array.asMutableArray)([_apm.AGENT_NAME, _apm.TELEMETRY_SDK_NAME, _apm.TELEMETRY_SDK_LANGUAGE, _apm.SERVICE_RUNTIME_NAME, _apm.CLOUD_PROVIDER, _apm.CLOUD_SERVICE_NAME]);
  const params = {
    terminate_after: 1,
    apm: {
      events: [_common.ProcessorEvent.error, _common.ProcessorEvent.transaction, _common.ProcessorEvent.metric]
    },
    track_total_hits: 1,
    size: 1,
    _source: [_apm.AGENT_NAME, _apm.SERVICE_RUNTIME_NAME, _apm.CLOUD_PROVIDER, _apm.CLOUD_SERVICE_NAME],
    query: {
      bool: {
        filter: [{
          term: {
            [_apm.SERVICE_NAME]: serviceName
          }
        }, ...(0, _server.rangeQuery)(start, end), {
          exists: {
            field: _apm.AGENT_NAME
          }
        }],
        should: [{
          exists: {
            field: _apm.SERVICE_RUNTIME_NAME
          }
        }, {
          exists: {
            field: _apm.CLOUD_PROVIDER
          }
        }, {
          exists: {
            field: _apm.CLOUD_SERVICE_NAME
          }
        }]
      }
    },
    fields,
    sort: {
      _score: {
        order: 'desc'
      }
    }
  };
  const response = await apmEventClient.search('get_service_agent_name', params);
  const hit = (0, _maybe.maybe)(response.hits.hits[0]);
  if (!hit) {
    return {};
  }
  const event = (0, _utils.accessKnownApmEventFields)(hit.fields);
  const serverlessType = (0, _serverless.getServerlessTypeFromCloudData)(event[_apm.CLOUD_PROVIDER], event[_apm.CLOUD_SERVICE_NAME]);
  return {
    agentName: event[_apm.AGENT_NAME],
    telemetrySdkName: event[_apm.TELEMETRY_SDK_NAME],
    telemetrySdkLanguage: event[_apm.TELEMETRY_SDK_LANGUAGE],
    runtimeName: event[_apm.SERVICE_RUNTIME_NAME],
    serverlessType
  };
}