"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.fetchUniqueKeysTool = fetchUniqueKeysTool;
var _tools = require("@langchain/core/tools");
var _messages = require("@langchain/core/messages");
var _langgraph = require("@langchain/langgraph");
var _zod = require("@kbn/zod");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const collectKeysWithSamples = (value, uniqueKeySamples, prefix = '') => {
  const storeSample = (key, sampleValue) => {
    const formattedSample = formatSample(sampleValue);
    const existingSample = uniqueKeySamples.get(key);
    if (typeof formattedSample === 'string') {
      if (formattedSample !== '') {
        uniqueKeySamples.set(key, formattedSample);
      } else if (existingSample === undefined) {
        uniqueKeySamples.set(key, '');
      }
      return;
    }
    if (existingSample === undefined) {
      uniqueKeySamples.set(key, formattedSample);
    }
  };
  if (value === null || value === undefined) {
    return;
  }
  if (Array.isArray(value)) {
    value.forEach(item => collectKeysWithSamples(item, uniqueKeySamples, prefix));
    return;
  }
  if (typeof value === 'object') {
    Object.entries(value).forEach(([key, val]) => {
      const nextPrefix = prefix ? `${prefix}.${key}` : key;
      storeSample(nextPrefix, val);
      collectKeysWithSamples(val, uniqueKeySamples, nextPrefix);
    });
    return;
  }
  if (prefix) {
    storeSample(prefix, value);
  }
};
const formatSample = value => {
  if (Array.isArray(value)) {
    return value.length > 0 ? formatSample(value[0]) : [];
  }
  if (typeof value === 'object' && value !== null) {
    return value;
  }
  return value !== null && value !== void 0 ? value : '';
};
function fetchUniqueKeysTool() {
  const schema = _zod.z.object({});
  return new _tools.DynamicStructuredTool({
    name: 'fetch_unique_keys',
    description: 'Retrieves processed documents from pipeline_generation_results stored in state. Use this to gather a subset ' + 'of recent pipeline outputs for downstream ECS mapping tasks.',
    schema,
    func: async (input, _runManager, config) => {
      var _pipelineDocs$docs, _config$toolCall2;
      const state = (0, _langgraph.getCurrentTaskInput)();
      const rawPipelineResults = state.pipeline_generation_results;
      const pipelineDocs = Array.isArray(rawPipelineResults) ? {
        docs: rawPipelineResults
      } : rawPipelineResults !== null && rawPipelineResults !== void 0 ? rawPipelineResults : {
        docs: []
      };
      const docs = (_pipelineDocs$docs = pipelineDocs.docs) !== null && _pipelineDocs$docs !== void 0 ? _pipelineDocs$docs : [];
      if (docs.length === 0) {
        var _config$toolCall;
        return new _langgraph.Command({
          update: {
            messages: [new _messages.ToolMessage({
              content: JSON.stringify({
                unique_keys: {}
              }),
              tool_call_id: config === null || config === void 0 ? void 0 : (_config$toolCall = config.toolCall) === null || _config$toolCall === void 0 ? void 0 : _config$toolCall.id
            })]
          }
        });
      }
      const uniqueKeySamples = new Map();
      docs.forEach(doc => {
        var _ref, _ref2, _doc$doc$_source, _doc$doc;
        const source = (_ref = (_ref2 = (_doc$doc$_source = doc === null || doc === void 0 ? void 0 : (_doc$doc = doc.doc) === null || _doc$doc === void 0 ? void 0 : _doc$doc._source) !== null && _doc$doc$_source !== void 0 ? _doc$doc$_source : doc === null || doc === void 0 ? void 0 : doc._source) !== null && _ref2 !== void 0 ? _ref2 : doc) !== null && _ref !== void 0 ? _ref : {};
        collectKeysWithSamples(source, uniqueKeySamples);
      });
      const uniqueKeysObject = Array.from(uniqueKeySamples.entries()).sort(([a], [b]) => a.localeCompare(b)).reduce((acc, [key, sample]) => {
        acc[key] = sample;
        return acc;
      }, {});
      return new _langgraph.Command({
        update: {
          messages: [new _messages.ToolMessage({
            content: JSON.stringify({
              unique_keys: uniqueKeysObject
            }),
            tool_call_id: config === null || config === void 0 ? void 0 : (_config$toolCall2 = config.toolCall) === null || _config$toolCall2 === void 0 ? void 0 : _config$toolCall2.id
          })]
        }
      });
    }
  });
}