"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerIntegrationRoutes = void 0;
var _server = require("@kbn/core/server");
var _zodHelpers = require("@kbn/zod-helpers");
var _common = require("../../common");
var _utils = require("./utils");
var _feature = require("../feature");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const registerIntegrationRoutes = (router, logger) => {
  getAllIntegrationsRoute(router, logger);
  getIntegrationByIdRoute(router, logger);
  createIntegrationRoute(router, logger);
};
exports.registerIntegrationRoutes = registerIntegrationRoutes;
const getAllIntegrationsRoute = (router, logger) => router.versioned.get({
  access: 'internal',
  path: '/api/automatic_import_v2/integrations',
  security: {
    authz: {
      requiredPrivileges: [`${_feature.AUTOMATIC_IMPORT_API_PRIVILEGES.READ}`]
    }
  }
}).addVersion({
  version: '1',
  validate: false
}, async (context, _, response) => {
  try {
    const automaticImportv2 = await context.automaticImportv2;
    const automaticImportService = automaticImportv2.automaticImportService;
    const integrationResponses = await automaticImportService.getAllIntegrations();
    const body = integrationResponses.map(integration => ({
      integrationId: integration.integrationId,
      title: integration.title,
      totalDataStreamCount: integration.dataStreams.length,
      successfulDataStreamCount: integration.dataStreams.filter(dataStream => dataStream.status === 'completed').length,
      status: integration.status
    }));
    return response.ok({
      body
    });
  } catch (err) {
    logger.error(`registerIntegrationRoutes: Caught error:`, err);
    const automaticImportResponse = (0, _utils.buildAutomaticImportResponse)(response);
    return automaticImportResponse.error({
      statusCode: 500,
      body: err
    });
  }
});
const getIntegrationByIdRoute = (router, logger) => router.versioned.get({
  access: 'internal',
  path: '/api/automatic_import_v2/integrations/{integration_id}',
  security: {
    authz: {
      requiredPrivileges: [`${_feature.AUTOMATIC_IMPORT_API_PRIVILEGES.READ}`]
    }
  }
}).addVersion({
  version: '1',
  validate: {
    request: {
      params: (0, _zodHelpers.buildRouteValidationWithZod)(_common.GetAutoImportIntegrationRequestParams)
    }
  }
}, async (context, request, response) => {
  try {
    const automaticImportv2 = await context.automaticImportv2;
    const automaticImportService = automaticImportv2.automaticImportService;
    const {
      integration_id: integrationId
    } = request.params;
    const integration = await automaticImportService.getIntegrationById(integrationId);
    const body = {
      integrationResponse: integration
    };
    return response.ok({
      body
    });
  } catch (err) {
    logger.error(`getIntegrationByIdRoute: Caught error:`, err);
    const automaticImportResponse = (0, _utils.buildAutomaticImportResponse)(response);
    const statusCode = _server.SavedObjectsErrorHelpers.isNotFoundError(err) ? 404 : 500;
    return automaticImportResponse.error({
      statusCode,
      body: err
    });
  }
});
const createIntegrationRoute = (router, logger) => router.versioned.put({
  access: 'internal',
  path: '/api/automatic_import_v2/integrations',
  security: {
    authz: {
      requiredPrivileges: [`${_feature.AUTOMATIC_IMPORT_API_PRIVILEGES.MANAGE}`]
    }
  }
}).addVersion({
  version: '1',
  validate: {
    request: {
      body: (0, _zodHelpers.buildRouteValidationWithZod)(_common.CreateAutoImportIntegrationRequestBody)
    }
  }
}, async (context, request, response) => {
  try {
    const automaticImportv2 = await context.automaticImportv2;
    const automaticImportService = automaticImportv2.automaticImportService;
    const integrationRequestBody = request.body;
    const authenticatedUser = await automaticImportv2.getCurrentUser();
    const integrationId = integrationRequestBody.integrationId;
    const integrationParams = {
      integrationId,
      title: integrationRequestBody.title,
      logo: integrationRequestBody.logo,
      description: integrationRequestBody.description
    };
    await automaticImportService.createIntegration({
      authenticatedUser,
      integrationParams
    });
    if (integrationRequestBody.dataStreams) {
      const dataStreamsParams = integrationRequestBody.dataStreams.map(dataStream => ({
        ...dataStream,
        integrationId,
        metadata: {
          createdAt: new Date().toISOString()
        }
      }));
      await Promise.all(dataStreamsParams.map(dataStreamParams => automaticImportService.createDataStream({
        authenticatedUser,
        dataStreamParams
      })));
    }
    const body = {
      integration_id: integrationId
    };
    return response.ok({
      body
    });
  } catch (err) {
    logger.error(`createIntegrationRoute: Caught error:`, err);
    const automaticImportResponse = (0, _utils.buildAutomaticImportResponse)(response);
    return automaticImportResponse.error({
      statusCode: 500,
      body: err
    });
  }
});