"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.AutomaticImportService = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _assert = _interopRequireDefault(require("assert"));
var _rxjs = require("rxjs");
var _index_service = require("./samples_index/index_service");
var _saved_objects_service = require("./saved_objects/saved_objects_service");
var _integration = require("./saved_objects/integration");
var _data_stream = require("./saved_objects/data_stream");
var _task_manager_service = require("./task_manager/task_manager_service");
var _constants = require("./saved_objects/constants");
var _task_manager = require("./task_manager");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class AutomaticImportService {
  constructor(logger, savedObjectsServiceSetup, taskManagerSetup) {
    (0, _defineProperty2.default)(this, "pluginStop$", void 0);
    (0, _defineProperty2.default)(this, "samplesIndexService", void 0);
    (0, _defineProperty2.default)(this, "savedObjectService", null);
    (0, _defineProperty2.default)(this, "logger", void 0);
    (0, _defineProperty2.default)(this, "savedObjectsServiceSetup", void 0);
    (0, _defineProperty2.default)(this, "taskManagerSetup", void 0);
    (0, _defineProperty2.default)(this, "taskManagerService", void 0);
    (0, _defineProperty2.default)(this, "processDataStreamWorkflow", async params => {
      // TODO: Implement the actual AI agent workflow here that uses AgentService
      // Will use params.integrationId and params.dataStreamId
    });
    this.pluginStop$ = new _rxjs.ReplaySubject(1);
    this.logger = logger;
    this.savedObjectsServiceSetup = savedObjectsServiceSetup;
    this.samplesIndexService = new _index_service.AutomaticImportSamplesIndexService(logger);
    this.savedObjectsServiceSetup.registerType(_integration.integrationSavedObjectType);
    this.savedObjectsServiceSetup.registerType(_data_stream.dataStreamSavedObjectType);
    this.taskManagerSetup = taskManagerSetup;
    this.taskManagerService = new _task_manager_service.TaskManagerService(this.logger, this.taskManagerSetup);
  }
  // Run initialize in the start phase of plugin
  async initialize(savedObjectsClient, taskManagerStart) {
    this.savedObjectService = new _saved_objects_service.AutomaticImportSavedObjectService(this.logger, savedObjectsClient);
    this.taskManagerService.initialize(taskManagerStart, {
      taskWorkflow: this.processDataStreamWorkflow
    });
  }
  async createIntegration(params) {
    (0, _assert.default)(this.savedObjectService, 'Saved Objects service not initialized.');
    const {
      authenticatedUser,
      integrationParams
    } = params;
    await this.savedObjectService.insertIntegration(integrationParams, authenticatedUser);
  }
  async updateIntegration(data, expectedVersion, versionUpdate, options) {
    if (!this.savedObjectService) {
      throw new Error('Saved Objects service not initialized.');
    }
    return this.savedObjectService.updateIntegration(data, expectedVersion, versionUpdate, options);
  }
  async getIntegrationById(integrationId) {
    if (!this.savedObjectService) {
      throw new Error('Saved Objects service not initialized.');
    }
    const integrationSO = await this.savedObjectService.getIntegration(integrationId);
    const dataStreamsSO = await this.savedObjectService.getAllDataStreams(integrationId);
    const dataStreamsResponses = dataStreamsSO.map(dataStream => ({
      dataStreamId: dataStream.data_stream_id,
      title: dataStream.title,
      description: dataStream.description,
      inputTypes: dataStream.input_types.map(type => ({
        name: type
      })),
      status: dataStream.job_info.status
    }));
    const integrationResponse = {
      integrationId: integrationSO.integration_id,
      title: integrationSO.metadata.title,
      logo: integrationSO.metadata.logo,
      description: integrationSO.metadata.description,
      status: integrationSO.status,
      dataStreams: dataStreamsResponses
    };
    return integrationResponse;
  }
  async getAllIntegrations() {
    if (!this.savedObjectService) {
      throw new Error('Saved Objects service not initialized.');
    }
    const integrations = await this.savedObjectService.getAllIntegrations();
    return Promise.all(integrations.map(async integration => {
      const dataStreams = await this.savedObjectService.getAllDataStreams(integration.integration_id);
      const dataStreamsResponses = dataStreams.map(dataStream => ({
        dataStreamId: dataStream.data_stream_id,
        title: dataStream.title,
        description: dataStream.description,
        inputTypes: dataStream.input_types.map(type => ({
          name: type
        })),
        status: dataStream.job_info.status
      }));
      return {
        integrationId: integration.integration_id,
        title: integration.metadata.title,
        logo: integration.metadata.logo,
        description: integration.metadata.description,
        status: integration.status,
        dataStreams: dataStreamsResponses
      };
    }));
  }
  async deleteIntegration(integrationId, options) {
    if (!this.savedObjectService) {
      throw new Error('Saved Objects service not initialized.');
    }
    return this.savedObjectService.deleteIntegration(integrationId, options);
  }
  async createDataStream(params) {
    (0, _assert.default)(this.savedObjectService, 'Saved Objects service not initialized.');
    const {
      authenticatedUser,
      dataStreamParams
    } = params;

    // Schedule the data stream creation Background task
    const dataStreamTaskParams = {
      integrationId: dataStreamParams.integrationId,
      dataStreamId: dataStreamParams.dataStreamId
    };
    const {
      taskId
    } = await this.taskManagerService.scheduleDataStreamCreationTask(dataStreamTaskParams);

    // Insert the data stream in saved object
    await this.savedObjectService.insertDataStream({
      ...dataStreamParams,
      jobInfo: {
        jobId: taskId,
        status: _constants.TASK_STATUSES.pending,
        jobType: _task_manager.DATA_STREAM_CREATION_TASK_TYPE
      }
    }, authenticatedUser);
  }
  async getDataStream(dataStreamId, integrationId) {
    if (!this.savedObjectService) {
      throw new Error('Saved Objects service not initialized.');
    }
    return this.savedObjectService.getDataStream(dataStreamId, integrationId);
  }
  async getAllDataStreams(integrationId) {
    if (!this.savedObjectService) {
      throw new Error('Saved Objects service not initialized.');
    }
    const dataStreams = await this.savedObjectService.getAllDataStreams(integrationId);
    return dataStreams.map(dataStream => ({
      data_stream_id: dataStream.data_stream_id,
      integration_id: integrationId,
      created_by: dataStream.created_by,
      title: dataStream.title,
      description: dataStream.description,
      input_types: dataStream.input_types,
      status: dataStream.job_info.status,
      job_info: dataStream.job_info,
      metadata: dataStream.metadata
    }));
  }
  async findAllDataStreamsByIntegrationId(integrationId) {
    if (!this.savedObjectService) {
      throw new Error('Saved Objects service not initialized.');
    }
    return this.savedObjectService.findAllDataStreamsByIntegrationId(integrationId);
  }
  async deleteDataStream(integrationId, dataStreamId, esClient, options) {
    (0, _assert.default)(this.savedObjectService, 'Saved Objects service not initialized.');
    // Remove the data stream creation task
    await this.taskManagerService.removeDataStreamCreationTask({
      integrationId,
      dataStreamId
    });
    // Delete the samples from the samples index
    await this.samplesIndexService.deleteSamplesForDataStream(integrationId, dataStreamId, esClient);
    // Delete the data stream from the saved objects
    await this.savedObjectService.deleteDataStream(integrationId, dataStreamId, options);
  }
  async addSamplesToDataStream(params) {
    return this.samplesIndexService.addSamplesToDataStream(params);
  }
  stop() {
    this.pluginStop$.next();
    this.pluginStop$.complete();
  }
}
exports.AutomaticImportService = AutomaticImportService;