"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.TaskManagerService = exports.DATA_STREAM_CREATION_TASK_TYPE = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _assert = _interopRequireDefault(require("assert"));
var _task = require("@kbn/task-manager-plugin/server/task");
var _constants = require("../constants");
var _constants2 = require("../saved_objects/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const DATA_STREAM_CREATION_TASK_TYPE = exports.DATA_STREAM_CREATION_TASK_TYPE = 'autoImport-dataStream-task';
class TaskManagerService {
  constructor(logger, taskManagerSetup) {
    (0, _defineProperty2.default)(this, "logger", void 0);
    (0, _defineProperty2.default)(this, "taskManager", null);
    (0, _defineProperty2.default)(this, "taskWorkflow", void 0);
    this.logger = logger.get('taskManagerService');

    // Register task definitions during setup phase
    taskManagerSetup.registerTaskDefinitions({
      [DATA_STREAM_CREATION_TASK_TYPE]: {
        title: 'Data Stream generation workflow',
        description: 'Executes long-running AI agent workflows for data stream generation',
        timeout: _constants.TASK_TIMEOUT_DURATION,
        maxAttempts: _constants.MAX_ATTEMPTS_AI_WORKFLOWS,
        cost: _task.TaskCost.Normal,
        priority: _task.TaskPriority.Normal,
        createTaskRunner: ({
          taskInstance
        }) => ({
          run: async () => this.runTask(taskInstance),
          cancel: async () => this.cancelTask(taskInstance)
        })
      }
    });
    this.logger.info(`Task definition "${DATA_STREAM_CREATION_TASK_TYPE}" registered`);
  }

  // for lifecycle start phase
  initialize(taskManager, options) {
    this.taskManager = taskManager;
    if (options !== null && options !== void 0 && options.taskWorkflow) {
      this.taskWorkflow = options.taskWorkflow;
    }
    this.logger.debug('TaskManagerService initialized');
  }
  async scheduleDataStreamCreationTask(params) {
    (0, _assert.default)(this.taskManager, 'TaskManager not initialized');
    const taskId = this.generateDataStreamTaskId(params);
    const taskInstance = await this.taskManager.schedule({
      id: taskId,
      taskType: DATA_STREAM_CREATION_TASK_TYPE,
      params,
      state: {
        task_status: _constants2.TASK_STATUSES.pending
      },
      scope: ['automaticImport']
    });
    this.logger.info(`Task scheduled: ${taskInstance.id}`);
    return {
      taskId: taskInstance.id
    };
  }
  async removeDataStreamCreationTask(dataStreamTaskParams) {
    (0, _assert.default)(this.taskManager, 'TaskManager not initialized');
    const taskId = this.generateDataStreamTaskId(dataStreamTaskParams);
    try {
      await this.taskManager.removeIfExists(taskId);
      this.logger.info(`Task deleted: ${taskId}`);
    } catch (error) {
      this.logger.error(`Failed to remove task ${taskId}:`, error);
    }
  }
  async getTaskStatus(taskId) {
    if (!this.taskManager) {
      throw new Error('TaskManager not initialized');
    }
    try {
      var _task$state;
      const task = await this.taskManager.get(taskId);
      return {
        task_status: (_task$state = task.state) === null || _task$state === void 0 ? void 0 : _task$state.task_status
      };
    } catch (error) {
      this.logger.error(`Failed to get task status for ${taskId}:`, error);
      throw new Error(`Task ${taskId} not found or inaccessible`);
    }
  }
  async runTask(taskInstance) {
    const {
      id: taskId,
      params
    } = taskInstance;
    this.logger.info(`Running task ${taskId}`, params);
    try {
      if (this.taskWorkflow) {
        const {
          integrationId,
          dataStreamId
        } = params;
        if (!integrationId || !dataStreamId) {
          throw new Error('Task params must include integrationId and dataStreamId');
        }
        await this.taskWorkflow({
          integrationId,
          dataStreamId
        });
        this.logger.debug(`Task ${taskId}: Workflow executed successfully`);
      } else {
        this.logger.warn(`Task ${taskId}: No workflow provided, task will complete without processing`);
      }
      this.logger.info(`Task ${taskId} completed successfully`);
      return {
        state: {
          task_status: _constants2.TASK_STATUSES.completed
        }
      };
    } catch (error) {
      this.logger.error(`Task ${taskId} failed:`, error);
      return {
        state: {
          task_status: _constants2.TASK_STATUSES.failed
        },
        error
      };
    }
  }
  async cancelTask(taskInstance) {
    // Cancel the AI task here
    this.logger.info(`Cancelling task ${taskInstance.id}`);
    return {
      state: {
        task_status: _constants2.TASK_STATUSES.cancelled
      }
    };
  }
  generateDataStreamTaskId(params) {
    return `data-stream-task-${params.integrationId}-${params.dataStreamId}`;
  }
}
exports.TaskManagerService = TaskManagerService;