"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useSubmitCase = void 0;
var _react = require("react");
var _use_post_case = require("../../containers/use_post_case");
var _use_post_push_to_service = require("../../containers/use_post_push_to_service");
var _use_create_attachments = require("../../containers/use_create_attachments");
var _use_cases_transactions = require("../../common/apm/use_cases_transactions");
var _use_application = require("../../common/lib/kibana/use_application");
var _use_bulk_post_observables = require("../../containers/use_bulk_post_observables");
var _use_attach_events_ebt = require("../../analytics/use_attach_events_ebt");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const useSubmitCase = ({
  attachments,
  observables,
  afterCaseCreated,
  onSuccess
}) => {
  const {
    appId
  } = (0, _use_application.useApplication)();
  const {
    mutateAsync: postCase,
    isLoading: isPostingCase
  } = (0, _use_post_case.usePostCase)();
  const {
    mutateAsync: createAttachments,
    isLoading: isCreatingAttachments
  } = (0, _use_create_attachments.useCreateAttachments)();
  const {
    mutateAsync: bulkPostObservables,
    isLoading: isPostingObservables
  } = (0, _use_bulk_post_observables.useBulkPostObservables)();
  const {
    mutateAsync: pushCaseToExternalService,
    isLoading: isPushingToExternalService
  } = (0, _use_post_push_to_service.usePostPushToService)();
  const {
    startTransaction
  } = (0, _use_cases_transactions.useCreateCaseWithAttachmentsTransaction)();
  const trackAttachEvents = (0, _use_attach_events_ebt.useAttachEventsEBT)();
  const submitCase = (0, _react.useCallback)(async (data, isValid) => {
    if (isValid) {
      startTransaction({
        appId,
        attachments
      });
      const theCase = await postCase({
        request: data
      });
      if (theCase && Array.isArray(attachments) && attachments.length > 0) {
        await createAttachments({
          caseId: theCase.id,
          caseOwner: theCase.owner,
          attachments
        });
        trackAttachEvents(window.location.pathname, attachments);
      }
      if (theCase && Array.isArray(observables) && observables.length > 0) {
        if (data.settings.extractObservables) {
          await bulkPostObservables({
            caseId: theCase.id,
            observables
          });
        }
      }
      if (afterCaseCreated && theCase) {
        await afterCaseCreated(theCase, createAttachments);
      }
      if (theCase !== null && theCase !== void 0 && theCase.id && data.connector.id !== 'none') {
        try {
          await pushCaseToExternalService({
            caseId: theCase.id,
            connector: data.connector
          });
        } catch (error) {
          // Catch the error but do not interrupt the flow.
          // The case has been created successfully at this point.
          // The only thing that failed was pushing to the external service.
          // Changes to the connector fields can be made later on by the user.
          // They will be notified about the connector failure.
        }
      }
      if (onSuccess && theCase) {
        onSuccess(theCase);
      }
    }
  }, [startTransaction, appId, attachments, postCase, observables, afterCaseCreated, onSuccess, createAttachments, trackAttachEvents, bulkPostObservables, pushCaseToExternalService]);
  return {
    submitCase,
    isSubmitting: isPostingCase || isCreatingAttachments || isPostingObservables || isPushingToExternalService
  };
};
exports.useSubmitCase = useSubmitCase;