"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.BackfillTaskRunner = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _server = require("@kbn/task-manager-plugin/server");
var _coreElasticsearchServerUtils = require("@kbn/core-elasticsearch-server-utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class BackfillTaskRunner {
  constructor({
    taskInstance,
    getESClient,
    logger,
    analyticsConfig
  }) {
    (0, _defineProperty2.default)(this, "sourceIndex", void 0);
    (0, _defineProperty2.default)(this, "destIndex", void 0);
    (0, _defineProperty2.default)(this, "sourceQuery", void 0);
    (0, _defineProperty2.default)(this, "getESClient", void 0);
    (0, _defineProperty2.default)(this, "logger", void 0);
    (0, _defineProperty2.default)(this, "errorSource", _server.TaskErrorSource.FRAMEWORK);
    (0, _defineProperty2.default)(this, "analyticsConfig", void 0);
    this.sourceIndex = taskInstance.params.sourceIndex;
    this.destIndex = taskInstance.params.destIndex;
    this.sourceQuery = taskInstance.params.sourceQuery;
    this.getESClient = getESClient;
    this.logger = logger;
    this.analyticsConfig = analyticsConfig;
  }
  async run() {
    if (!this.analyticsConfig.index.enabled) {
      this.logDebug('Analytics index is disabled, skipping backfill task.');
      return;
    }
    const esClient = await this.getESClient();
    try {
      await this.waitForDestIndex(esClient);
      await this.backfillReindex(esClient);
      return {
        // one time only tasks get deleted so this state is not enough
        // for the periodic tasks to know the backfill was complete
        state: {} // ?
      };
    } catch (e) {
      if ((0, _coreElasticsearchServerUtils.isRetryableEsClientError)(e)) {
        (0, _server.throwRetryableError)((0, _server.createTaskRunError)(new Error(this.getErrorMessage(e.message)), this.errorSource), true);
      }
      this.logger.error(`[${this.destIndex}] Backfill reindex failed. Error: ${e.message}`, {
        tags: ['cai-backfill', 'cai-backfill-error', this.destIndex]
      });
      (0, _server.throwUnrecoverableError)((0, _server.createTaskRunError)(e, this.errorSource));
    }
  }
  async cancel() {}
  async backfillReindex(esClient) {
    const painlessScript = await this.getPainlessScript(esClient);
    if (painlessScript.found) {
      this.logDebug(`Reindexing from ${this.sourceIndex} to ${this.destIndex}.`);
      const painlessScriptId = await this.getPainlessScriptId(esClient);
      await esClient.reindex({
        source: {
          index: this.sourceIndex,
          query: this.sourceQuery
        },
        dest: {
          index: this.destIndex
        },
        script: {
          id: painlessScriptId
        },
        /** If `true`, the request refreshes affected shards to make this operation visible to search. */
        refresh: true,
        /** We do not wait for the es reindex operation to be completed. */
        wait_for_completion: false
      });
    } else {
      throw (0, _server.createTaskRunError)(new Error(this.getErrorMessage('Painless script not found.')), this.errorSource);
    }
  }
  async getPainlessScript(esClient) {
    const painlessScriptId = await this.getPainlessScriptId(esClient);
    this.logDebug(`Getting painless script with id ${painlessScriptId}.`);
    return esClient.getScript({
      id: painlessScriptId
    });
  }
  async getPainlessScriptId(esClient) {
    var _currentMapping$this$;
    const currentMapping = await this.getCurrentMapping(esClient);
    const painlessScriptId = (_currentMapping$this$ = currentMapping[this.destIndex].mappings._meta) === null || _currentMapping$this$ === void 0 ? void 0 : _currentMapping$this$.painless_script_id;
    if (!painlessScriptId) {
      throw (0, _server.createTaskRunError)(new Error(this.getErrorMessage('Painless script id missing from mapping.')), this.errorSource);
    }
    return painlessScriptId;
  }
  async getCurrentMapping(esClient) {
    this.logDebug('Getting index mapping.');
    return esClient.indices.getMapping({
      index: this.destIndex
    });
  }
  async waitForDestIndex(esClient) {
    this.logDebug('Checking index availability.');
    return esClient.cluster.health({
      index: this.destIndex,
      wait_for_status: 'green',
      timeout: '30s'
    });
  }
  logDebug(message) {
    this.logger.debug(`[${this.destIndex}] ${message}`, {
      tags: ['cai-backfill', this.destIndex]
    });
  }
  getErrorMessage(message) {
    const errorMessage = `[${this.destIndex}] Backfill reindex failed. Error: ${message}`;
    this.logger.error(errorMessage, {
      tags: ['cai-backfill', 'cai-backfill-error', this.destIndex]
    });
    return errorMessage;
  }
}
exports.BackfillTaskRunner = BackfillTaskRunner;