"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.find = void 0;
var _lodash = require("lodash");
var _boom = _interopRequireDefault(require("@hapi/boom"));
var _api = require("../../../common/types/api");
var _runtime_types = require("../../common/runtime_types");
var _error = require("../../common/error");
var _utils = require("../../common/utils");
var _utils2 = require("../utils");
var _authorization = require("../../authorization");
var _constants = require("../../common/constants");
var _validators = require("./validators");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Retrieves a case and optionally its comments via saved objects find.
 *
 * @ignore
 */
const find = async (params, clientArgs, casesClient) => {
  const {
    services: {
      caseService,
      licensingService
    },
    authorization,
    logger,
    savedObjectsSerializer,
    spaceId
  } = clientArgs;
  try {
    const paramArgs = (0, _runtime_types.decodeWithExcessOrThrow)(_api.CasesFindRequestWithCustomFieldsRt)(params);
    const configArgs = paramArgs.owner ? {
      owner: paramArgs.owner
    } : {};
    const configurations = await casesClient.configure.get(configArgs);
    const customFieldsConfiguration = configurations.map(config => config.customFields).flat();

    /**
     * Assign users to a case is only available to Platinum+
     */

    if (!(0, _lodash.isEmpty)(paramArgs.assignees)) {
      const hasPlatinumLicenseOrGreater = await licensingService.isAtLeastPlatinum();
      if (!hasPlatinumLicenseOrGreater) {
        throw _boom.default.forbidden('In order to filter cases by assignees, you must be subscribed to an Elastic Platinum license');
      }
      licensingService.notifyUsage(_constants.LICENSING_CASE_ASSIGNMENT_FEATURE);
    }

    /**
     * Validate custom fields
     */
    if (paramArgs !== null && paramArgs !== void 0 && paramArgs.customFields && !(0, _lodash.isEmpty)(paramArgs === null || paramArgs === void 0 ? void 0 : paramArgs.customFields)) {
      /**
       * throw error if params has customFields and no owner
       */

      const isValidArray = (0, _lodash.isArray)(paramArgs.owner) && (!paramArgs.owner.length || paramArgs.owner.length > 1 || (0, _lodash.isEmpty)(paramArgs.owner[0]));
      if (!paramArgs.owner || isValidArray) {
        throw _boom.default.badRequest('Owner must be provided. Multiple owners are not supported.');
      }
      (0, _validators.validateSearchCasesCustomFields)({
        customFieldsConfiguration,
        customFields: paramArgs.customFields
      });
    }
    const {
      filter: authorizationFilter,
      ensureSavedObjectsAreAuthorized
    } = await authorization.getAuthorizationFilter(_authorization.Operations.findCases);
    const options = {
      tags: paramArgs.tags,
      reporters: paramArgs.reporters,
      sortField: paramArgs.sortField,
      status: paramArgs.status,
      severity: paramArgs.severity,
      owner: paramArgs.owner,
      from: paramArgs.from,
      to: paramArgs.to,
      assignees: paramArgs.assignees,
      category: paramArgs.category,
      customFields: paramArgs.customFields
    };
    const statusStatsOptions = (0, _utils2.constructQueryOptions)({
      ...options,
      status: undefined,
      customFieldsConfiguration,
      authorizationFilter
    });
    const caseQueryOptions = (0, _utils2.constructQueryOptions)({
      ...options,
      customFieldsConfiguration,
      authorizationFilter
    });
    const caseSearch = (0, _utils2.constructSearch)(paramArgs.search, spaceId, savedObjectsSerializer);
    const [cases, statusStats] = await Promise.all([caseService.findCasesGroupedByID({
      caseOptions: {
        ...paramArgs,
        ...caseQueryOptions,
        ...caseSearch,
        searchFields: (0, _utils.asArray)(paramArgs.searchFields)
      }
    }), caseService.getCaseStatusStats({
      searchOptions: statusStatsOptions
    })]);
    ensureSavedObjectsAreAuthorized([...cases.casesMap.values()]);
    const res = (0, _utils.transformCases)({
      casesMap: cases.casesMap,
      page: cases.page,
      perPage: cases.perPage,
      total: cases.total,
      countOpenCases: statusStats.open,
      countInProgressCases: statusStats['in-progress'],
      countClosedCases: statusStats.closed
    });
    return (0, _runtime_types.decodeOrThrow)(_api.CasesFindResponseRt)(res);
  } catch (error) {
    throw (0, _error.createCaseError)({
      message: `Failed to find cases: ${JSON.stringify(params)}: ${error}`,
      error,
      logger
    });
  }
};
exports.find = find;