"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.CasesConnector = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _boom = _interopRequireDefault(require("@hapi/boom"));
var _server = require("@kbn/actions-plugin/server");
var _common = require("../../../common");
var _schema = require("./schema");
var _cases_oracle_service = require("./cases_oracle_service");
var _cases_service = require("./cases_service");
var _cases_connector_error = require("./cases_connector_error");
var _cases_connector_executor = require("./cases_connector_executor");
var _cases_connector_retry_service = require("./cases_connector_retry_service");
var _retry_service = require("../../common/retry_service");
var _constants = require("../../../common/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class CasesConnector extends _server.SubActionConnector {
  constructor({
    connectorParams,
    casesParams
  }) {
    super(connectorParams);
    (0, _defineProperty2.default)(this, "casesService", void 0);
    (0, _defineProperty2.default)(this, "retryService", void 0);
    (0, _defineProperty2.default)(this, "casesParams", void 0);
    this.casesService = new _cases_service.CasesService();

    /**
     * We should wait at least 5ms before retrying and no more that 2sec
     */
    const backOffFactory = (0, _retry_service.fullJitterBackoffFactory)({
      baseDelay: 5,
      maxBackoffTime: 2000
    });
    this.retryService = new _cases_connector_retry_service.CasesConnectorRetryService(this.logger, backOffFactory);
    this.casesParams = casesParams;
    this.registerSubActions();
  }
  registerSubActions() {
    this.registerSubAction({
      name: _constants.CASES_CONNECTOR_SUB_ACTION.RUN,
      method: 'run',
      schema: _schema.ZCasesConnectorRunParamsSchema
    });
  }

  /**
   * Method is not needed for the Case Connector.
   * The function throws an error as a reminder to
   * implement it if we need it in the future.
   */
  getResponseErrorMessage() {
    throw new Error('Method not implemented.');
  }
  async run(params) {
    if (!this.kibanaRequest) {
      const error = new _cases_connector_error.CasesConnectorError('Kibana request is not defined', 400);
      this.handleError(error);
    }
    if (params.alerts.length === 0) {
      this.logDebugCurrentState('start', '[CasesConnector][_run] No alerts. Skipping execution.', params);
      return;
    }
    await this.retryService.retryWithBackoff(() => this._run(params));
  }
  async _run(params) {
    try {
      /**
       * The case connector will throw an error if the Kibana request
       * is not define before executing the _run method
       */
      const kibanaRequest = this.kibanaRequest;
      const casesClient = await this.casesParams.getCasesClient(kibanaRequest);
      const savedObjectsClient = await this.casesParams.getUnsecuredSavedObjectsClient(kibanaRequest, [..._common.SAVED_OBJECT_TYPES, _constants.CASE_RULES_SAVED_OBJECT]);
      const spaceId = this.casesParams.getSpaceId(kibanaRequest);
      const casesOracleService = new _cases_oracle_service.CasesOracleService({
        logger: this.logger,
        savedObjectsClient
      });
      const connectorExecutor = new _cases_connector_executor.CasesConnectorExecutor({
        logger: this.logger,
        casesOracleService,
        casesService: this.casesService,
        casesClient,
        spaceId
      });
      this.logDebugCurrentState('start', '[CasesConnector][_run] Executing case connector', params);
      await connectorExecutor.execute(params);
      this.logDebugCurrentState('success', '[CasesConnector][_run] Execution of case connector succeeded', params);
    } catch (error) {
      this.handleError(error);
    } finally {
      this.logDebugCurrentState('end', '[CasesConnector][_run] Execution of case connector ended', params);
    }
  }
  handleError(error) {
    if ((0, _cases_connector_error.isCasesConnectorError)(error)) {
      const userError = (0, _cases_connector_error.createTaskUserError)(error);
      this.logError(userError);
      throw userError;
    }
    if ((0, _cases_connector_error.isCasesClientError)(error)) {
      const caseConnectorError = new _cases_connector_error.CasesConnectorError(error.message, error.boomify().output.statusCode);
      const userError = (0, _cases_connector_error.createTaskUserError)(caseConnectorError);
      this.logError(userError);
      throw userError;
    }
    if (_boom.default.isBoom(error)) {
      const caseConnectorError = new _cases_connector_error.CasesConnectorError(`${error.output.payload.error}: ${error.output.payload.message}`, error.output.statusCode);
      const userError = (0, _cases_connector_error.createTaskUserError)(caseConnectorError);
      this.logError(userError);
      throw userError;
    }
    const caseConnectorError = new _cases_connector_error.CasesConnectorError(error.message, 500);
    this.logError(caseConnectorError);
    throw caseConnectorError;
  }
  logDebugCurrentState(state, message, params) {
    const alertIds = params.alerts.map(({
      _id
    }) => _id);
    this.logger.debug(`[CasesConnector][_run] ${message}`, {
      labels: {
        ruleId: params.rule.id,
        groupingBy: params.groupingBy,
        totalAlerts: params.alerts.length,
        timeWindow: params.timeWindow,
        reopenClosedCases: params.reopenClosedCases,
        owner: params.owner
      },
      tags: [`cases-connector:${state}`, params.rule.id, ...alertIds]
    });
  }
  logError(error) {
    this.logger.error(`[CasesConnector][run] Execution of case connector failed. Message: ${error.message}. Status code: ${error.statusCode}`, {
      error: {
        stack_trace: error.stack,
        code: error.statusCode.toString(),
        type: 'CasesConnectorError'
      }
    });
  }
}
exports.CasesConnector = CasesConnector;