"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.CasesOracleService = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _server = require("@kbn/alerting-plugin/server");
var _constants = require("../../../common/constants");
var _error = require("../../common/error");
var _constants2 = require("./constants");
var _crypto_service = require("./crypto_service");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class CasesOracleService {
  constructor({
    logger,
    savedObjectsClient
  }) {
    (0, _defineProperty2.default)(this, "logger", void 0);
    (0, _defineProperty2.default)(this, "savedObjectsClient", void 0);
    (0, _defineProperty2.default)(this, "cryptoService", void 0);
    (0, _defineProperty2.default)(this, "getRecordResponse", oracleRecord => {
      var _oracleRecord$version;
      return {
        id: oracleRecord.id,
        version: (_oracleRecord$version = oracleRecord.version) !== null && _oracleRecord$version !== void 0 ? _oracleRecord$version : '',
        counter: oracleRecord.attributes.counter,
        grouping: oracleRecord.attributes.grouping,
        rules: oracleRecord.attributes.rules,
        createdAt: oracleRecord.attributes.createdAt,
        updatedAt: oracleRecord.attributes.updatedAt
      };
    });
    this.logger = logger;
    this.savedObjectsClient = savedObjectsClient;
    this.cryptoService = new _crypto_service.CryptoService();
  }
  getRecordId({
    ruleId,
    spaceId,
    owner,
    grouping
  }) {
    if (grouping == null && ruleId == null) {
      throw new Error('ruleID or grouping is required');
    }
    const payload = [ruleId, spaceId, owner, this.cryptoService.stringifyDeterministically(grouping)].filter(Boolean).join(':');
    return this.cryptoService.getHash(payload);
  }
  async getRecord(recordId) {
    this.logger.debug(`Getting oracle record with ID: ${recordId}`, {
      tags: ['cases-oracle-service', 'getRecord', recordId]
    });
    const oracleRecord = await this.savedObjectsClient.get(_constants.CASE_RULES_SAVED_OBJECT, recordId);
    return this.getRecordResponse(oracleRecord);
  }
  async bulkGetRecords(ids) {
    this.logger.debug(`Getting oracle records with IDs: ${ids}`, {
      tags: ['cases-oracle-service', 'bulkGetRecords', ...ids]
    });
    if (ids.length === 0) {
      return [];
    }
    const oracleRecords = await this.savedObjectsClient.bulkGet(ids.map(id => ({
      id,
      type: _constants.CASE_RULES_SAVED_OBJECT
    })));
    return this.getBulkRecordsResponse(oracleRecords);
  }
  async createRecord(recordId, payload) {
    this.logger.debug(`Creating oracle record with ID: ${recordId}`, {
      tags: ['cases-oracle-service', 'createRecord', recordId]
    });
    const oracleRecord = await this.savedObjectsClient.create(_constants.CASE_RULES_SAVED_OBJECT, this.getCreateRecordAttributes(payload), {
      id: recordId,
      references: this.getCreateRecordReferences(payload)
    });
    return this.getRecordResponse(oracleRecord);
  }
  async bulkCreateRecord(records) {
    const recordIds = records.map(record => record.recordId);
    this.logger.debug(`Creating oracle record with ID: ${recordIds}`, {
      tags: ['cases-oracle-service', 'bulkCreateRecord', ...recordIds]
    });
    if (records.length === 0) {
      return [];
    }
    const req = records.map(record => ({
      id: record.recordId,
      type: _constants.CASE_RULES_SAVED_OBJECT,
      attributes: this.getCreateRecordAttributes(record.payload),
      references: this.getCreateRecordReferences(record.payload)
    }));
    const oracleRecords = await this.savedObjectsClient.bulkCreate(req);
    return this.getBulkRecordsResponse(oracleRecords);
  }
  async increaseCounter(recordId) {
    var _oracleRecord$referen;
    const {
      id: _,
      version,
      ...record
    } = await this.getRecord(recordId);
    const newCounter = record.counter + 1;
    this.logger.debug(`Increasing the counter of oracle record with ID: ${recordId} from ${record.counter} to ${newCounter}`, {
      tags: ['cases-oracle-service', 'increaseCounter', recordId]
    });
    const oracleRecord = await this.savedObjectsClient.update(_constants.CASE_RULES_SAVED_OBJECT, recordId, {
      counter: newCounter
    }, {
      version
    });
    return this.getRecordResponse({
      ...oracleRecord,
      attributes: {
        ...record,
        counter: newCounter
      },
      references: (_oracleRecord$referen = oracleRecord.references) !== null && _oracleRecord$referen !== void 0 ? _oracleRecord$referen : []
    });
  }
  async bulkUpdateRecord(records) {
    const recordIds = records.map(record => record.recordId);
    this.logger.debug(`Updating oracle record with ID: ${recordIds}`, {
      tags: ['cases-oracle-service', 'bulkUpdateRecord', ...recordIds]
    });
    if (records.length === 0) {
      return [];
    }
    const req = records.map(record => ({
      id: record.recordId,
      type: _constants.CASE_RULES_SAVED_OBJECT,
      version: record.version,
      attributes: {
        ...record.payload,
        updatedAt: new Date().toISOString()
      }
    }));
    const oracleRecords = await this.savedObjectsClient.bulkUpdate(req);
    return this.getBulkRecordsResponse(oracleRecords);
  }
  getBulkRecordsResponse(oracleRecords) {
    return oracleRecords.saved_objects.map(oracleRecord => {
      if ((0, _error.isSOError)(oracleRecord)) {
        return this.getErrorResponse(oracleRecord.id, oracleRecord.error);
      }
      return this.getRecordResponse(oracleRecord);
    });
  }
  getErrorResponse(id, error) {
    if ((0, _error.isSODecoratedError)(error)) {
      return {
        id,
        error: error.output.payload.error,
        message: error.output.payload.message,
        statusCode: error.output.statusCode
      };
    }
    return {
      id,
      error: error.error,
      message: error.message,
      statusCode: error.statusCode
    };
  }
  getCreateRecordAttributes({
    rules,
    grouping
  }) {
    return {
      counter: _constants2.INITIAL_ORACLE_RECORD_COUNTER,
      rules,
      grouping,
      createdAt: new Date().toISOString(),
      updatedAt: null
    };
  }
  getCreateRecordReferences({
    rules,
    grouping
  }) {
    const references = [];
    for (const rule of rules) {
      references.push({
        id: rule.id,
        type: _server.RULE_SAVED_OBJECT_TYPE,
        name: `associated-${_server.RULE_SAVED_OBJECT_TYPE}`
      });
    }
    return references;
  }
}
exports.CasesOracleService = CasesOracleService;