"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.mergeSearchQuery = exports.convertFindQueryParams = exports.constructSearchQuery = exports.DEFAULT_CASE_SEARCH_FIELDS = exports.DEFAULT_CASE_NESTED_FIELDS = exports.DEFAULT_ATTACHMENT_SEARCH_FIELDS = void 0;
var _constants = require("../../../common/constants");
var _api = require("../../routes/api");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const DEFAULT_CASE_SEARCH_FIELDS = exports.DEFAULT_CASE_SEARCH_FIELDS = [`${_constants.CASE_SAVED_OBJECT}.title`, `${_constants.CASE_SAVED_OBJECT}.description`, `${_constants.CASE_SAVED_OBJECT}.incremental_id.text`];
const DEFAULT_CASE_NESTED_FIELDS = exports.DEFAULT_CASE_NESTED_FIELDS = [`${_constants.CASE_SAVED_OBJECT}.observables.value`, `${_constants.CASE_SAVED_OBJECT}.customFields.value`];
const DEFAULT_ATTACHMENT_SEARCH_FIELDS = exports.DEFAULT_ATTACHMENT_SEARCH_FIELDS = [`${_constants.CASE_COMMENT_SAVED_OBJECT}.alertId`, `${_constants.CASE_COMMENT_SAVED_OBJECT}.eventId`, `${_constants.CASE_COMMENT_SAVED_OBJECT}.comment`];
const mergeSearchQuery = (searchQuery, filterQuery) => {
  if (!searchQuery && !filterQuery) {
    return undefined;
  }
  if (!searchQuery) {
    return filterQuery;
  }
  if (!filterQuery) {
    return searchQuery;
  }

  // If filterQuery doesn't have bool, wrap both queries in a bool query
  if (!filterQuery.bool) {
    return {
      bool: {
        filter: [searchQuery, filterQuery]
      }
    };
  }

  // Normalize existing filter to an array
  const existingFilter = filterQuery.bool.filter ? Array.isArray(filterQuery.bool.filter) ? filterQuery.bool.filter : [filterQuery.bool.filter] : [];

  // Merge searchQuery into the existing bool query's filter array
  return {
    bool: {
      ...filterQuery.bool,
      filter: [searchQuery, ...existingFilter]
    }
  };
};

/**
 * Constructs a search query for cases.
 * When search is provided, it will be used to search for cases in
 *** Default search fields.
 *** Nested fields.
 *** Case IDs.
 * When caseIds is provided, it will be used to filter cases by case IDs.
 */
exports.mergeSearchQuery = mergeSearchQuery;
const constructSearchQuery = ({
  search,
  searchFields,
  caseIds
}) => {
  const caseSearchFields = searchFields === null || searchFields === void 0 ? void 0 : searchFields.filter(field => DEFAULT_CASE_SEARCH_FIELDS.includes(field));
  const nestedFields = searchFields === null || searchFields === void 0 ? void 0 : searchFields.filter(field => DEFAULT_CASE_NESTED_FIELDS.includes(field));
  if (!search && !caseIds.length) {
    return undefined;
  }
  const shouldClauses = [];
  if (search) {
    // support search by case ID
    shouldClauses.push({
      term: {
        [`_id`]: `${_constants.CASE_SAVED_OBJECT}:${search}`
      }
    });
    if (caseSearchFields !== null && caseSearchFields !== void 0 && caseSearchFields.length) {
      shouldClauses.push({
        multi_match: {
          query: search,
          fields: caseSearchFields
        }
      });
    }
    if (nestedFields !== null && nestedFields !== void 0 && nestedFields.length) {
      nestedFields.forEach(field => {
        const path = field.split('.').slice(0, -1).join('.');
        if (searchFields !== null && searchFields !== void 0 && searchFields.includes(field) && path) {
          shouldClauses.push({
            nested: {
              path,
              query: {
                term: {
                  [field]: {
                    value: search,
                    case_insensitive: true
                  }
                }
              }
            }
          });
        }
      });
    }
  }
  if (caseIds.length > 0) {
    shouldClauses.push({
      terms: {
        [`_id`]: caseIds.map(id => `${_constants.CASE_SAVED_OBJECT}:${id}`)
      }
    });
  }
  return {
    bool: {
      should: shouldClauses,
      minimum_should_match: 1
    }
  };
};
exports.constructSearchQuery = constructSearchQuery;
const convertFindQueryParams = findOptions => {
  const {
    page,
    perPage,
    sortField,
    sortOrder,
    fields
  } = findOptions;
  const from = ((page !== null && page !== void 0 ? page : 1) - 1) * (perPage !== null && perPage !== void 0 ? perPage : _api.DEFAULT_PER_PAGE);
  const size = perPage !== null && perPage !== void 0 ? perPage : _api.DEFAULT_PER_PAGE;
  const sort = sortField && sortOrder ? {
    [`${_constants.CASE_SAVED_OBJECT}.${sortField}`]: {
      order: sortOrder
    }
  } : undefined;
  return {
    from,
    size,
    sort,
    fields
  };
};
exports.convertFindQueryParams = convertFindQueryParams;