"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.processWithAlertsByOwner = exports.getUniqueAlertCommentsCountQuery = exports.getTotalWithMaxObservables = exports.getTotalStats = exports.getSolutionValues = exports.getSolutionStats = exports.getReferencesAggregationQuery = exports.getOnlyConnectorsFilter = exports.getOnlyAlertsCommentsFilter = exports.getObservablesTotalsByType = exports.getMaxBucketOnCaseAggregationQuery = exports.getCustomFieldsTelemetry = exports.getCountsFromBuckets = exports.getCountsAndMaxData = exports.getCountsAndMaxAlertsData = exports.getCountsAggregationQuery = exports.getConnectorsCardinalityAggregationQuery = exports.getBucketFromAggregation = exports.getAttachmentsFrameworkStats = exports.getAlertsMaxBucketOnCaseAggregationQuery = exports.getAlertsCountsFromBuckets = exports.getAlertsCountsAggregationQuery = exports.getAggregationsBuckets = exports.findValueInBuckets = void 0;
var _lodash = require("lodash");
var _constants = require("../../../common/constants");
var _observables = require("../../../common/constants/observables");
var _utils = require("../../client/utils");
var _case = require("../../common/types/case");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getCountsAggregationQuery = savedObjectType => ({
  counts: {
    date_range: {
      field: `${savedObjectType}.attributes.created_at`,
      format: 'dd/MM/yyyy',
      ranges: [{
        from: 'now-1d',
        to: 'now'
      }, {
        from: 'now-1w',
        to: 'now'
      }, {
        from: 'now-1M',
        to: 'now'
      }]
    }
  }
});
exports.getCountsAggregationQuery = getCountsAggregationQuery;
const getAlertsCountsAggregationQuery = () => ({
  counts: {
    date_range: {
      field: `${_constants.CASE_COMMENT_SAVED_OBJECT}.attributes.created_at`,
      format: 'dd/MM/yyyy',
      ranges: [{
        from: 'now-1d',
        to: 'now'
      }, {
        from: 'now-1w',
        to: 'now'
      }, {
        from: 'now-1M',
        to: 'now'
      }]
    },
    aggregations: {
      topAlertsPerBucket: {
        cardinality: {
          field: `${_constants.CASE_COMMENT_SAVED_OBJECT}.attributes.alertId`
        }
      }
    }
  }
});
exports.getAlertsCountsAggregationQuery = getAlertsCountsAggregationQuery;
const getMaxBucketOnCaseAggregationQuery = savedObjectType => ({
  references: {
    nested: {
      path: `${savedObjectType}.references`
    },
    aggregations: {
      cases: {
        filter: {
          term: {
            [`${savedObjectType}.references.type`]: _constants.CASE_SAVED_OBJECT
          }
        },
        aggregations: {
          ids: {
            terms: {
              field: `${savedObjectType}.references.id`
            }
          },
          max: {
            max_bucket: {
              buckets_path: 'ids._count'
            }
          }
        }
      }
    }
  }
});
exports.getMaxBucketOnCaseAggregationQuery = getMaxBucketOnCaseAggregationQuery;
const getAlertsMaxBucketOnCaseAggregationQuery = () => ({
  references: {
    nested: {
      path: `${_constants.CASE_COMMENT_SAVED_OBJECT}.references`
    },
    aggregations: {
      cases: {
        filter: {
          term: {
            [`${_constants.CASE_COMMENT_SAVED_OBJECT}.references.type`]: _constants.CASE_SAVED_OBJECT
          }
        },
        aggregations: {
          ids: {
            terms: {
              field: `${_constants.CASE_COMMENT_SAVED_OBJECT}.references.id`
            },
            aggregations: {
              reverse: {
                reverse_nested: {},
                aggregations: {
                  topAlerts: {
                    cardinality: {
                      field: `${_constants.CASE_COMMENT_SAVED_OBJECT}.attributes.alertId`
                    }
                  }
                }
              }
            }
          },
          max: {
            max_bucket: {
              buckets_path: 'ids>reverse.topAlerts'
            }
          }
        }
      }
    }
  }
});
exports.getAlertsMaxBucketOnCaseAggregationQuery = getAlertsMaxBucketOnCaseAggregationQuery;
const getUniqueAlertCommentsCountQuery = () => ({
  uniqueAlertCommentsCount: {
    cardinality: {
      field: `${_constants.CASE_COMMENT_SAVED_OBJECT}.attributes.alertId`
    }
  }
});
exports.getUniqueAlertCommentsCountQuery = getUniqueAlertCommentsCountQuery;
const getReferencesAggregationQuery = ({
  savedObjectType,
  referenceType,
  agg = 'terms'
}) => ({
  references: {
    nested: {
      path: `${savedObjectType}.references`
    },
    aggregations: {
      referenceType: {
        filter: {
          term: {
            [`${savedObjectType}.references.type`]: referenceType
          }
        },
        aggregations: {
          referenceAgg: {
            [agg]: {
              field: `${savedObjectType}.references.id`
            }
          }
        }
      }
    }
  }
});
exports.getReferencesAggregationQuery = getReferencesAggregationQuery;
const getConnectorsCardinalityAggregationQuery = () => getReferencesAggregationQuery({
  savedObjectType: _constants.CASE_USER_ACTION_SAVED_OBJECT,
  referenceType: 'action',
  agg: 'cardinality'
});
exports.getConnectorsCardinalityAggregationQuery = getConnectorsCardinalityAggregationQuery;
const getCountsFromBuckets = buckets => {
  var _buckets$2$doc_count, _buckets$, _buckets$1$doc_count, _buckets$2, _buckets$0$doc_count, _buckets$3;
  return {
    daily: (_buckets$2$doc_count = buckets === null || buckets === void 0 ? void 0 : (_buckets$ = buckets[2]) === null || _buckets$ === void 0 ? void 0 : _buckets$.doc_count) !== null && _buckets$2$doc_count !== void 0 ? _buckets$2$doc_count : 0,
    weekly: (_buckets$1$doc_count = buckets === null || buckets === void 0 ? void 0 : (_buckets$2 = buckets[1]) === null || _buckets$2 === void 0 ? void 0 : _buckets$2.doc_count) !== null && _buckets$1$doc_count !== void 0 ? _buckets$1$doc_count : 0,
    monthly: (_buckets$0$doc_count = buckets === null || buckets === void 0 ? void 0 : (_buckets$3 = buckets[0]) === null || _buckets$3 === void 0 ? void 0 : _buckets$3.doc_count) !== null && _buckets$0$doc_count !== void 0 ? _buckets$0$doc_count : 0
  };
};
exports.getCountsFromBuckets = getCountsFromBuckets;
const getAlertsCountsFromBuckets = buckets => {
  var _buckets$2$topAlertsP, _buckets$4, _buckets$4$topAlertsP, _buckets$1$topAlertsP, _buckets$5, _buckets$5$topAlertsP, _buckets$0$topAlertsP, _buckets$6, _buckets$6$topAlertsP;
  return {
    daily: (_buckets$2$topAlertsP = buckets === null || buckets === void 0 ? void 0 : (_buckets$4 = buckets[2]) === null || _buckets$4 === void 0 ? void 0 : (_buckets$4$topAlertsP = _buckets$4.topAlertsPerBucket) === null || _buckets$4$topAlertsP === void 0 ? void 0 : _buckets$4$topAlertsP.value) !== null && _buckets$2$topAlertsP !== void 0 ? _buckets$2$topAlertsP : 0,
    weekly: (_buckets$1$topAlertsP = buckets === null || buckets === void 0 ? void 0 : (_buckets$5 = buckets[1]) === null || _buckets$5 === void 0 ? void 0 : (_buckets$5$topAlertsP = _buckets$5.topAlertsPerBucket) === null || _buckets$5$topAlertsP === void 0 ? void 0 : _buckets$5$topAlertsP.value) !== null && _buckets$1$topAlertsP !== void 0 ? _buckets$1$topAlertsP : 0,
    monthly: (_buckets$0$topAlertsP = buckets === null || buckets === void 0 ? void 0 : (_buckets$6 = buckets[0]) === null || _buckets$6 === void 0 ? void 0 : (_buckets$6$topAlertsP = _buckets$6.topAlertsPerBucket) === null || _buckets$6$topAlertsP === void 0 ? void 0 : _buckets$6$topAlertsP.value) !== null && _buckets$0$topAlertsP !== void 0 ? _buckets$0$topAlertsP : 0
  };
};
exports.getAlertsCountsFromBuckets = getAlertsCountsFromBuckets;
const getObservablesTotalsByType = observablesAggs => {
  var _observablesAggs$byDe;
  const result = {
    manual: {
      default: 0,
      custom: 0
    },
    auto: {
      default: 0,
      custom: 0
    },
    total: 0
  };
  if (!observablesAggs || !((_observablesAggs$byDe = observablesAggs.byDescription) !== null && _observablesAggs$byDe !== void 0 && _observablesAggs$byDe.buckets)) {
    return result;
  }
  observablesAggs.byDescription.buckets.forEach(bucket => {
    const description = bucket.key;
    bucket.byType.buckets.forEach(typeBucket => {
      const type = _observables.OBSERVABLE_TYPES_BUILTIN_KEYS.includes(typeBucket.key) ? 'default' : 'custom';
      const count = typeBucket.doc_count;
      if (description === 'Auto extract observables') {
        result.auto[type] += count;
      } else {
        result.manual[type] += count;
      }
      result.total += count;
    });
  });
  return result;
};
exports.getObservablesTotalsByType = getObservablesTotalsByType;
const getTotalWithMaxObservables = totalWithMaxObservablesAgg => {
  if (!totalWithMaxObservablesAgg || totalWithMaxObservablesAgg.length === 0) {
    return 0;
  }

  // Sum doc_count for all buckets where key (total_observables value) >= 50
  return totalWithMaxObservablesAgg.reduce((sum, bucket) => {
    var _bucket$doc_count;
    const key = typeof bucket.key === 'number' ? bucket.key : Number(bucket.key);
    return key >= _constants.MAX_OBSERVABLES_PER_CASE ? sum + ((_bucket$doc_count = bucket.doc_count) !== null && _bucket$doc_count !== void 0 ? _bucket$doc_count : 0) : sum;
  }, 0);
};
exports.getTotalWithMaxObservables = getTotalWithMaxObservables;
const getCountsAndMaxAlertsData = async ({
  savedObjectsClient
}) => {
  const filter = getOnlyAlertsCommentsFilter();
  const res = await savedObjectsClient.find({
    page: 0,
    perPage: 0,
    filter,
    type: _constants.CASE_COMMENT_SAVED_OBJECT,
    namespaces: ['*'],
    aggs: {
      by_owner: {
        terms: {
          field: `${_constants.CASE_COMMENT_SAVED_OBJECT}.attributes.owner`,
          size: 3,
          include: ['securitySolution', 'observability', 'cases']
        },
        aggs: {
          ...getAlertsCountsAggregationQuery(),
          ...getAlertsMaxBucketOnCaseAggregationQuery(),
          ...getUniqueAlertCommentsCountQuery()
        }
      }
    }
  });
  const sec = getSolutionStats('securitySolution', res === null || res === void 0 ? void 0 : res.aggregations);
  const obs = getSolutionStats('observability', res === null || res === void 0 ? void 0 : res.aggregations);
  const main = getSolutionStats('cases', res === null || res === void 0 ? void 0 : res.aggregations);
  const all = getTotalStats(res === null || res === void 0 ? void 0 : res.aggregations);
  return {
    all,
    sec,
    obs,
    main
  };
};
exports.getCountsAndMaxAlertsData = getCountsAndMaxAlertsData;
const getSolutionStats = (owner, countsAndMaxAlertsAggRes) => {
  var _countsAndMaxAlertsAg, _countsAndMaxAlertsAg2, _bucket$uniqueAlertCo, _bucket$uniqueAlertCo2, _bucket$counts$bucket, _bucket$counts, _bucket$references$ca, _bucket$references, _bucket$references$ca2, _bucket$references$ca3;
  const bucket = countsAndMaxAlertsAggRes === null || countsAndMaxAlertsAggRes === void 0 ? void 0 : (_countsAndMaxAlertsAg = countsAndMaxAlertsAggRes.by_owner) === null || _countsAndMaxAlertsAg === void 0 ? void 0 : (_countsAndMaxAlertsAg2 = _countsAndMaxAlertsAg.buckets) === null || _countsAndMaxAlertsAg2 === void 0 ? void 0 : _countsAndMaxAlertsAg2.find(b => (b === null || b === void 0 ? void 0 : b.key) === owner);
  if (!bucket) {
    return {
      total: 0,
      daily: 0,
      weekly: 0,
      monthly: 0,
      maxOnACase: 0
    };
  }
  return {
    total: (_bucket$uniqueAlertCo = bucket === null || bucket === void 0 ? void 0 : (_bucket$uniqueAlertCo2 = bucket.uniqueAlertCommentsCount) === null || _bucket$uniqueAlertCo2 === void 0 ? void 0 : _bucket$uniqueAlertCo2.value) !== null && _bucket$uniqueAlertCo !== void 0 ? _bucket$uniqueAlertCo : 0,
    ...getAlertsCountsFromBuckets((_bucket$counts$bucket = bucket === null || bucket === void 0 ? void 0 : (_bucket$counts = bucket.counts) === null || _bucket$counts === void 0 ? void 0 : _bucket$counts.buckets) !== null && _bucket$counts$bucket !== void 0 ? _bucket$counts$bucket : []),
    maxOnACase: (_bucket$references$ca = bucket === null || bucket === void 0 ? void 0 : (_bucket$references = bucket.references) === null || _bucket$references === void 0 ? void 0 : (_bucket$references$ca2 = _bucket$references.cases) === null || _bucket$references$ca2 === void 0 ? void 0 : (_bucket$references$ca3 = _bucket$references$ca2.max) === null || _bucket$references$ca3 === void 0 ? void 0 : _bucket$references$ca3.value) !== null && _bucket$references$ca !== void 0 ? _bucket$references$ca : 0
  };
};
exports.getSolutionStats = getSolutionStats;
const getTotalStats = countsAndMaxAlertsAggRes => {
  var _countsAndMaxAlertsAg3, _countsAndMaxAlertsAg4;
  const buckets = (_countsAndMaxAlertsAg3 = countsAndMaxAlertsAggRes === null || countsAndMaxAlertsAggRes === void 0 ? void 0 : (_countsAndMaxAlertsAg4 = countsAndMaxAlertsAggRes.by_owner) === null || _countsAndMaxAlertsAg4 === void 0 ? void 0 : _countsAndMaxAlertsAg4.buckets) !== null && _countsAndMaxAlertsAg3 !== void 0 ? _countsAndMaxAlertsAg3 : [];
  return buckets.reduce((acc, bucket) => {
    var _bucket$uniqueAlertCo3, _bucket$uniqueAlertCo4, _bucket$counts$bucket2, _bucket$counts2, _bucket$references$ca4, _bucket$references2, _bucket$references2$c, _bucket$references2$c2;
    acc.total += (_bucket$uniqueAlertCo3 = bucket === null || bucket === void 0 ? void 0 : (_bucket$uniqueAlertCo4 = bucket.uniqueAlertCommentsCount) === null || _bucket$uniqueAlertCo4 === void 0 ? void 0 : _bucket$uniqueAlertCo4.value) !== null && _bucket$uniqueAlertCo3 !== void 0 ? _bucket$uniqueAlertCo3 : 0;
    const counts = getAlertsCountsFromBuckets((_bucket$counts$bucket2 = bucket === null || bucket === void 0 ? void 0 : (_bucket$counts2 = bucket.counts) === null || _bucket$counts2 === void 0 ? void 0 : _bucket$counts2.buckets) !== null && _bucket$counts$bucket2 !== void 0 ? _bucket$counts$bucket2 : []);
    acc.daily += counts.daily;
    acc.weekly += counts.weekly;
    acc.monthly += counts.monthly;
    const maxCaseVal = (_bucket$references$ca4 = bucket === null || bucket === void 0 ? void 0 : (_bucket$references2 = bucket.references) === null || _bucket$references2 === void 0 ? void 0 : (_bucket$references2$c = _bucket$references2.cases) === null || _bucket$references2$c === void 0 ? void 0 : (_bucket$references2$c2 = _bucket$references2$c.max) === null || _bucket$references2$c2 === void 0 ? void 0 : _bucket$references2$c2.value) !== null && _bucket$references$ca4 !== void 0 ? _bucket$references$ca4 : 0;
    if (maxCaseVal > acc.maxOnACase) {
      acc.maxOnACase = maxCaseVal;
    }
    return acc;
  }, {
    total: 0,
    daily: 0,
    weekly: 0,
    monthly: 0,
    maxOnACase: 0
  });
};
exports.getTotalStats = getTotalStats;
const getCountsAndMaxData = async ({
  savedObjectsClient,
  savedObjectType,
  filter
}) => {
  var _res$aggregations$cou, _res$aggregations, _res$aggregations$cou2, _res$aggregations$ref, _res$aggregations2, _res$aggregations2$re, _res$aggregations2$re2, _res$aggregations2$re3;
  const res = await savedObjectsClient.find({
    page: 0,
    perPage: 0,
    filter,
    type: savedObjectType,
    namespaces: ['*'],
    aggs: {
      ...getCountsAggregationQuery(savedObjectType),
      ...getMaxBucketOnCaseAggregationQuery(savedObjectType)
    }
  });
  const countsBuckets = (_res$aggregations$cou = (_res$aggregations = res.aggregations) === null || _res$aggregations === void 0 ? void 0 : (_res$aggregations$cou2 = _res$aggregations.counts) === null || _res$aggregations$cou2 === void 0 ? void 0 : _res$aggregations$cou2.buckets) !== null && _res$aggregations$cou !== void 0 ? _res$aggregations$cou : [];
  const maxOnACase = (_res$aggregations$ref = (_res$aggregations2 = res.aggregations) === null || _res$aggregations2 === void 0 ? void 0 : (_res$aggregations2$re = _res$aggregations2.references) === null || _res$aggregations2$re === void 0 ? void 0 : (_res$aggregations2$re2 = _res$aggregations2$re.cases) === null || _res$aggregations2$re2 === void 0 ? void 0 : (_res$aggregations2$re3 = _res$aggregations2$re2.max) === null || _res$aggregations2$re3 === void 0 ? void 0 : _res$aggregations2$re3.value) !== null && _res$aggregations$ref !== void 0 ? _res$aggregations$ref : 0;
  return {
    all: {
      total: res.total,
      ...getCountsFromBuckets(countsBuckets),
      maxOnACase
    }
  };
};
exports.getCountsAndMaxData = getCountsAndMaxData;
const getBucketFromAggregation = ({
  aggs,
  key
}) => {
  var _get;
  return (_get = (0, _lodash.get)(aggs, `${key}.buckets`)) !== null && _get !== void 0 ? _get : [];
};
exports.getBucketFromAggregation = getBucketFromAggregation;
const getSolutionValues = ({
  caseAggregations,
  attachmentAggregations,
  filesAggregations,
  casesTotalWithAlerts,
  owner
}) => {
  var _caseAggregations$own, _caseAggregations$own2, _caseAggregations$own3, _caseAggregations$own4, _caseAggregations$own5, _caseAggregations$own6, _caseAggregations$own7;
  const aggregationsBuckets = getAggregationsBuckets({
    aggs: caseAggregations,
    keys: ['totalsByOwner', 'securitySolution.counts', 'observability.counts', 'cases.counts', `${owner}.status`]
  });
  const totalCasesForOwner = findValueInBuckets(aggregationsBuckets.totalsByOwner, owner);
  const attachmentsAggsForOwner = attachmentAggregations === null || attachmentAggregations === void 0 ? void 0 : attachmentAggregations[owner];
  const fileAttachmentsForOwner = filesAggregations === null || filesAggregations === void 0 ? void 0 : filesAggregations[owner];
  const totalWithAlerts = processWithAlertsByOwner(casesTotalWithAlerts);
  return {
    total: totalCasesForOwner,
    ...getCountsFromBuckets(aggregationsBuckets[`${owner}.counts`]),
    status: {
      open: findValueInBuckets(aggregationsBuckets[`${owner}.status`], _case.CasePersistedStatus.OPEN),
      inProgress: findValueInBuckets(aggregationsBuckets[`${owner}.status`], _case.CasePersistedStatus.IN_PROGRESS),
      closed: findValueInBuckets(aggregationsBuckets[`${owner}.status`], _case.CasePersistedStatus.CLOSED)
    },
    ...getAttachmentsFrameworkStats({
      attachmentAggregations: attachmentsAggsForOwner,
      filesAggregations: fileAttachmentsForOwner,
      totalCasesForOwner
    }),
    observables: getObservablesTotalsByType(caseAggregations === null || caseAggregations === void 0 ? void 0 : (_caseAggregations$own = caseAggregations[owner]) === null || _caseAggregations$own === void 0 ? void 0 : _caseAggregations$own.observables),
    totalWithMaxObservables: getTotalWithMaxObservables((_caseAggregations$own2 = caseAggregations === null || caseAggregations === void 0 ? void 0 : (_caseAggregations$own3 = caseAggregations[owner]) === null || _caseAggregations$own3 === void 0 ? void 0 : (_caseAggregations$own4 = _caseAggregations$own3.totalWithMaxObservables) === null || _caseAggregations$own4 === void 0 ? void 0 : _caseAggregations$own4.buckets) !== null && _caseAggregations$own2 !== void 0 ? _caseAggregations$own2 : []),
    totalWithAlerts: totalWithAlerts[owner],
    assignees: {
      total: (_caseAggregations$own5 = caseAggregations === null || caseAggregations === void 0 ? void 0 : caseAggregations[owner].totalAssignees.value) !== null && _caseAggregations$own5 !== void 0 ? _caseAggregations$own5 : 0,
      totalWithZero: (_caseAggregations$own6 = caseAggregations === null || caseAggregations === void 0 ? void 0 : caseAggregations[owner].assigneeFilters.buckets.zero.doc_count) !== null && _caseAggregations$own6 !== void 0 ? _caseAggregations$own6 : 0,
      totalWithAtLeastOne: (_caseAggregations$own7 = caseAggregations === null || caseAggregations === void 0 ? void 0 : caseAggregations[owner].assigneeFilters.buckets.atLeastOne.doc_count) !== null && _caseAggregations$own7 !== void 0 ? _caseAggregations$own7 : 0
    }
  };
};
exports.getSolutionValues = getSolutionValues;
const getCustomFieldsTelemetry = customFields => {
  var _customFields$length;
  const customFiledTypes = {};
  const totalsByType = customFields === null || customFields === void 0 ? void 0 : customFields.reduce((a, c) => {
    if (c !== null && c !== void 0 && c.type) {
      var _customFiledTypes$c$t;
      Object.assign(customFiledTypes, {
        [c.type]: ((_customFiledTypes$c$t = customFiledTypes[c.type]) !== null && _customFiledTypes$c$t !== void 0 ? _customFiledTypes$c$t : 0) + 1
      });
    }
    return customFiledTypes;
  }, {});
  const allRequiredCustomFields = customFields === null || customFields === void 0 ? void 0 : customFields.filter(field => field === null || field === void 0 ? void 0 : field.required).length;
  return {
    totalsByType: totalsByType !== null && totalsByType !== void 0 ? totalsByType : {},
    totals: (_customFields$length = customFields === null || customFields === void 0 ? void 0 : customFields.length) !== null && _customFields$length !== void 0 ? _customFields$length : 0,
    required: allRequiredCustomFields !== null && allRequiredCustomFields !== void 0 ? allRequiredCustomFields : 0
  };
};
exports.getCustomFieldsTelemetry = getCustomFieldsTelemetry;
const findValueInBuckets = (buckets, value) => {
  var _buckets$find$doc_cou, _buckets$find;
  return (_buckets$find$doc_cou = (_buckets$find = buckets.find(({
    key
  }) => key === value)) === null || _buckets$find === void 0 ? void 0 : _buckets$find.doc_count) !== null && _buckets$find$doc_cou !== void 0 ? _buckets$find$doc_cou : 0;
};
exports.findValueInBuckets = findValueInBuckets;
const getAggregationsBuckets = ({
  aggs,
  keys
}) => keys.reduce((acc, key) => {
  acc[key] = getBucketFromAggregation({
    aggs,
    key
  });
  return acc;
}, {});
exports.getAggregationsBuckets = getAggregationsBuckets;
const getAttachmentsFrameworkStats = ({
  attachmentAggregations,
  filesAggregations,
  totalCasesForOwner
}) => {
  if (!attachmentAggregations) {
    return emptyAttachmentFramework();
  }
  const averageFileSize = getAverageFileSize(filesAggregations);
  const topMimeTypes = filesAggregations === null || filesAggregations === void 0 ? void 0 : filesAggregations.topMimeTypes;
  return {
    attachmentFramework: {
      externalAttachments: getAttachmentRegistryStats(attachmentAggregations.externalReferenceTypes, totalCasesForOwner),
      persistableAttachments: getAttachmentRegistryStats(attachmentAggregations.persistableReferenceTypes, totalCasesForOwner),
      files: getFileAttachmentStats({
        registryResults: attachmentAggregations.externalReferenceTypes,
        averageFileSize,
        totalCasesForOwner,
        topMimeTypes
      })
    }
  };
};
exports.getAttachmentsFrameworkStats = getAttachmentsFrameworkStats;
const getAverageFileSize = filesAggregations => {
  var _filesAggregations$av;
  if ((filesAggregations === null || filesAggregations === void 0 ? void 0 : (_filesAggregations$av = filesAggregations.averageSize) === null || _filesAggregations$av === void 0 ? void 0 : _filesAggregations$av.value) == null) {
    return 0;
  }
  return Math.round(filesAggregations.averageSize.value);
};
const getAttachmentRegistryStats = (registryResults, totalCasesForOwner) => {
  const stats = [];
  for (const bucket of registryResults.buckets) {
    const commonFields = {
      average: calculateTypePerCaseAverage(bucket.doc_count, totalCasesForOwner),
      maxOnACase: bucket.references.cases.max.value,
      total: bucket.doc_count
    };
    stats.push({
      type: bucket.key,
      ...commonFields
    });
  }
  return stats;
};
const calculateTypePerCaseAverage = (typeDocCount, totalCases) => {
  if (typeDocCount == null || totalCases === 0) {
    return 0;
  }
  return Math.round(typeDocCount / totalCases);
};
const getFileAttachmentStats = ({
  registryResults,
  averageFileSize,
  totalCasesForOwner,
  topMimeTypes
}) => {
  var _topMimeTypes$buckets, _fileBucket$reference, _fileBucket$doc_count;
  const fileBucket = registryResults.buckets.find(bucket => bucket.key === _constants.FILE_ATTACHMENT_TYPE);
  const mimeTypes = (_topMimeTypes$buckets = topMimeTypes === null || topMimeTypes === void 0 ? void 0 : topMimeTypes.buckets.map(mimeType => ({
    count: mimeType.doc_count,
    name: mimeType.key
  }))) !== null && _topMimeTypes$buckets !== void 0 ? _topMimeTypes$buckets : [];
  return {
    averageSize: averageFileSize !== null && averageFileSize !== void 0 ? averageFileSize : 0,
    average: calculateTypePerCaseAverage(fileBucket === null || fileBucket === void 0 ? void 0 : fileBucket.doc_count, totalCasesForOwner),
    maxOnACase: (_fileBucket$reference = fileBucket === null || fileBucket === void 0 ? void 0 : fileBucket.references.cases.max.value) !== null && _fileBucket$reference !== void 0 ? _fileBucket$reference : 0,
    total: (_fileBucket$doc_count = fileBucket === null || fileBucket === void 0 ? void 0 : fileBucket.doc_count) !== null && _fileBucket$doc_count !== void 0 ? _fileBucket$doc_count : 0,
    topMimeTypes: mimeTypes
  };
};
const getOnlyAlertsCommentsFilter = () => (0, _utils.buildFilter)({
  filters: ['alert'],
  field: 'type',
  operator: 'or',
  type: _constants.CASE_COMMENT_SAVED_OBJECT
});
exports.getOnlyAlertsCommentsFilter = getOnlyAlertsCommentsFilter;
const getOnlyConnectorsFilter = () => (0, _utils.buildFilter)({
  filters: ['connector'],
  field: 'type',
  operator: 'or',
  type: _constants.CASE_USER_ACTION_SAVED_OBJECT
});
exports.getOnlyConnectorsFilter = getOnlyConnectorsFilter;
const emptyAttachmentFramework = () => ({
  attachmentFramework: {
    persistableAttachments: [],
    externalAttachments: [],
    files: emptyFileAttachment()
  }
});
const emptyFileAttachment = () => ({
  average: 0,
  averageSize: 0,
  maxOnACase: 0,
  total: 0,
  topMimeTypes: []
});
const processWithAlertsByOwner = aggregations => {
  const result = {
    securitySolution: 0,
    observability: 0,
    cases: 0
  };
  if (aggregations) {
    var _aggregations$by_owne;
    (_aggregations$by_owne = aggregations.by_owner) === null || _aggregations$by_owne === void 0 ? void 0 : _aggregations$by_owne.buckets.forEach(item => {
      result[item.key] = item.references.referenceType.referenceAgg.value;
    });
  }
  return result;
};
exports.processWithAlertsByOwner = processWithAlertsByOwner;