"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.updateBenchmarkRulesStates = exports.setRulesStates = exports.muteDetectionRules = exports.getDetectionRules = exports.getDetectionRuleIdsToDisable = exports.getBenchmarkRules = void 0;
var _cloudSecurityPostureCommon = require("@kbn/cloud-security-posture-common");
var _detection_rules = require("../../../../common/utils/detection_rules");
var _constants = require("../../../../common/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getDetectionRuleIdsToDisable = async detectionRules => {
  const idsToDisable = detectionRules.map(detectionRule => {
    return detectionRule.data.map(data => data.id);
  }).flat();
  return idsToDisable;
};
exports.getDetectionRuleIdsToDisable = getDetectionRuleIdsToDisable;
const disableDetectionRules = async (detectionRulesClient, detectionRules) => {
  const detectionRulesIdsToDisable = await getDetectionRuleIdsToDisable(detectionRules);
  if (!detectionRulesIdsToDisable.length) return [];
  const uniqueDetectionRulesIdsToDisable = [...new Set(detectionRulesIdsToDisable)]; // Prevent muting the same rule twice.
  await detectionRulesClient.bulkDisableRules({
    ids: uniqueDetectionRulesIdsToDisable
  });
  return uniqueDetectionRulesIdsToDisable;
};
const getDetectionRules = async (detectionRulesClient, rulesTags) => {
  const detectionRules = Promise.all(rulesTags.map(async ruleTags => {
    return detectionRulesClient.find({
      excludeFromPublicApi: false,
      options: {
        filter: (0, _detection_rules.convertRuleTagsToMatchAllKQL)(ruleTags),
        searchFields: ['tags'],
        page: 1,
        perPage: 100 // Disable up to 100 detection rules per benchmark rule at a time
      }
    });
  }));
  return detectionRules;
};
exports.getDetectionRules = getDetectionRules;
const getBenchmarkRules = async (soClient, ruleIds) => {
  const bulkGetObject = ruleIds.map(ruleId => ({
    id: ruleId,
    type: _constants.CSP_BENCHMARK_RULE_SAVED_OBJECT_TYPE
  }));
  const cspBenchmarkRulesSo = await soClient.bulkGet(bulkGetObject);
  const benchmarkRules = cspBenchmarkRulesSo.saved_objects.map(cspBenchmarkRule => cspBenchmarkRule.attributes);
  return benchmarkRules;
};
exports.getBenchmarkRules = getBenchmarkRules;
const muteDetectionRules = async (soClient, detectionRulesClient, rulesIds) => {
  const benchmarkRules = await getBenchmarkRules(soClient, rulesIds);
  if (benchmarkRules.includes(undefined)) {
    throw new Error('At least one of the provided benchmark rule IDs does not exist');
  }
  const benchmarkRulesTags = benchmarkRules.map(benchmarkRule => (0, _cloudSecurityPostureCommon.generateBenchmarkRuleTags)(benchmarkRule.metadata));
  const detectionRules = await getDetectionRules(detectionRulesClient, benchmarkRulesTags);
  const disabledDetectionRules = await disableDetectionRules(detectionRulesClient, detectionRules);
  return disabledDetectionRules;
};
exports.muteDetectionRules = muteDetectionRules;
const updateBenchmarkRulesStates = async (encryptedSoClient, newRulesStates) => {
  if (!Object.keys(newRulesStates).length) {
    return {};
  }
  const response = await encryptedSoClient.update(_constants.INTERNAL_CSP_SETTINGS_SAVED_OBJECT_TYPE, _constants.INTERNAL_CSP_SETTINGS_SAVED_OBJECT_ID, {
    rules: newRulesStates
  },
  // if there is no saved object yet, insert a new SO
  {
    upsert: {
      rules: newRulesStates
    }
  });
  return response.attributes.rules || {};
};
exports.updateBenchmarkRulesStates = updateBenchmarkRulesStates;
const setRulesStates = (ruleIds, state, rulesToUpdate) => {
  const rulesStates = {};
  ruleIds.forEach((ruleId, index) => {
    const benchmarkRule = rulesToUpdate[index];
    rulesStates[ruleId] = {
      muted: state,
      benchmark_id: benchmarkRule.benchmark_id,
      benchmark_version: benchmarkRule.benchmark_version,
      rule_number: benchmarkRule.rule_number,
      rule_id: benchmarkRule.rule_id
    };
  });
  return rulesStates;
};
exports.setRulesStates = setRulesStates;