"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.hasAuthSchema = exports.authTypeSchema = exports.SecretConfigurationSchemaValidation = exports.SecretConfigurationSchema = exports.SecretConfiguration = exports.AuthConfiguration = void 0;
var _i18n = require("@kbn/i18n");
var _zod = require("@kbn/zod");
var _constants = require("../constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const authTypeSchema = exports.authTypeSchema = _zod.z.union([_zod.z.literal(_constants.AuthType.Basic), _zod.z.literal(_constants.AuthType.SSL), _zod.z.literal(_constants.AuthType.OAuth2ClientCredentials), _zod.z.literal(null)]).default(_constants.AuthType.Basic).optional();
const hasAuthSchema = exports.hasAuthSchema = _zod.z.boolean().default(true);
const HeadersSchema = _zod.z.record(_zod.z.string(), _zod.z.string());
const AuthConfiguration = exports.AuthConfiguration = {
  hasAuth: hasAuthSchema,
  authType: authTypeSchema,
  certType: _zod.z.enum([_constants.SSLCertType.CRT, _constants.SSLCertType.PFX]).optional(),
  ca: _zod.z.string().optional(),
  verificationMode: _zod.z.enum(['none', 'certificate', 'full']).optional(),
  accessTokenUrl: _zod.z.string().optional(),
  clientId: _zod.z.string().optional(),
  scope: _zod.z.string().optional(),
  additionalFields: _zod.z.string().nullish()
};
const SecretConfiguration = exports.SecretConfiguration = {
  user: _zod.z.string().nullable().default(null),
  password: _zod.z.string().nullable().default(null),
  crt: _zod.z.string().nullable().default(null),
  key: _zod.z.string().nullable().default(null),
  pfx: _zod.z.string().nullable().default(null),
  clientSecret: _zod.z.string().nullable().default(null),
  secretHeaders: HeadersSchema.nullable().default(null)
};
const SecretConfigurationSchemaValidation = exports.SecretConfigurationSchemaValidation = {
  validate: secrets => {
    // user and password must be set together (or not at all)
    if (!secrets.password && !secrets.user && !secrets.crt && !secrets.key && !secrets.pfx && !secrets.clientSecret) {
      return;
    }
    if (secrets.password && secrets.user && !secrets.crt && !secrets.key && !secrets.pfx && !secrets.clientSecret) {
      return;
    }
    if (secrets.crt && secrets.key && !secrets.user && !secrets.pfx && !secrets.clientSecret) {
      return;
    }
    if (!secrets.crt && !secrets.key && !secrets.user && secrets.pfx && !secrets.clientSecret) {
      return;
    }
    if (!secrets.password && !secrets.user && !secrets.crt && !secrets.key && !secrets.pfx && secrets.clientSecret) {
      return;
    }
    return _i18n.i18n.translate('xpack.stackConnectors.webhook.invalidSecrets', {
      defaultMessage: 'must specify one of the following schemas: user and password; crt and key (with optional password); pfx (with optional password); or clientSecret (for OAuth2)'
    });
  }
};
const SecretConfigurationSchema = exports.SecretConfigurationSchema = _zod.z.object(SecretConfiguration).strict().superRefine((secrets, ctx) => {
  const errorMessage = SecretConfigurationSchemaValidation.validate(secrets);
  if (errorMessage) {
    ctx.addIssue({
      code: _zod.z.ZodIssueCode.custom,
      message: errorMessage
    });
  }
});