"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.TestConnectorRequestSchema = exports.MCPConnectorSecretsSchema = exports.MCPConnectorConfigSchema = exports.MCPAuthType = exports.ListToolsRequestSchema = exports.CallToolRequestSchema = void 0;
var _zod = require("@kbn/zod");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * Authentication types supported by the MCP connector.
 */
const MCPAuthType = exports.MCPAuthType = {
  None: 'none',
  Bearer: 'bearer',
  ApiKey: 'apiKey',
  Basic: 'basic'
};

/**
 * Schema for MCP connector configuration.
 *
 * Flat structure following standard Kibana connector patterns.
 */
const MCPConnectorConfigSchema = exports.MCPConnectorConfigSchema = _zod.z.object({
  /**
   * The URL of the MCP server endpoint.
   */
  serverUrl: _zod.z.string(),
  /**
   * Whether authentication is required. Defaults to true.
   */
  hasAuth: _zod.z.boolean().default(true),
  /**
   * Authentication type to use when hasAuth is true.
   */
  authType: _zod.z.enum([MCPAuthType.None, MCPAuthType.Bearer, MCPAuthType.ApiKey, MCPAuthType.Basic]).optional(),
  /**
   * Custom header name for API key authentication.
   * Defaults to 'X-API-Key' if not specified.
   * Only used when authType is 'apiKey'.
   */
  apiKeyHeaderName: _zod.z.string().min(1).optional(),
  /**
   * Non-sensitive HTTP headers to include in requests.
   */
  headers: _zod.z.record(_zod.z.string(), _zod.z.string()).optional()
});

/**
 * Schema for MCP connector secrets.
 *
 * Flat structure with optional fields based on auth type.
 */
const MCPConnectorSecretsSchema = exports.MCPConnectorSecretsSchema = _zod.z.object({
  /**
   * Bearer token for 'bearer' auth type.
   */
  token: _zod.z.string().optional(),
  /**
   * API key for 'apiKey' auth type.
   */
  apiKey: _zod.z.string().optional(),
  /**
   * Username for 'basic' auth type.
   */
  user: _zod.z.string().optional(),
  /**
   * Password for 'basic' auth type.
   */
  password: _zod.z.string().optional(),
  /**
   * Sensitive HTTP headers to include in requests.
   */
  secretHeaders: _zod.z.record(_zod.z.string(), _zod.z.string()).optional()
});

// Sub-action schemas

const TestConnectorRequestSchema = exports.TestConnectorRequestSchema = _zod.z.object({}).strict();
const ListToolsRequestSchema = exports.ListToolsRequestSchema = _zod.z.object({
  forceRefresh: _zod.z.boolean().optional()
});
const CallToolRequestSchema = exports.CallToolRequestSchema = _zod.z.object({
  name: _zod.z.string(),
  arguments: _zod.z.record(_zod.z.string(), _zod.z.any()).optional()
});