"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.SecretsSchema = exports.ParamsSchema = exports.ConfigSchema = void 0;
var _i18n = require("@kbn/i18n");
var _zod = require("@kbn/zod");
var _moment = _interopRequireDefault(require("moment"));
var _utils = require("../../common/utils");
var _constants = require("../constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const configSchemaProps = {
  apiUrl: _zod.z.string().nullable().default(null)
};
const ConfigSchema = exports.ConfigSchema = _zod.z.object(configSchemaProps).strict();
const SecretsSchema = exports.SecretsSchema = _zod.z.object({
  routingKey: _zod.z.string()
}).strict();
const EventActionSchema = _zod.z.enum([_constants.EVENT_ACTION_TRIGGER, _constants.EVENT_ACTION_RESOLVE, _constants.EVENT_ACTION_ACKNOWLEDGE]);
const PayloadSeveritySchema = _zod.z.enum(['critical', 'error', 'warning', 'info']);
const LinksSchema = _zod.z.array(_zod.z.object({
  href: _zod.z.string(),
  text: _zod.z.string()
}).strict());
const customDetailsSchema = _zod.z.record(_zod.z.string(), _zod.z.any());
const ParamsSchema = exports.ParamsSchema = _zod.z.object({
  eventAction: EventActionSchema.optional(),
  dedupKey: _zod.z.string().max(255).optional(),
  summary: _zod.z.string().max(1024).optional(),
  source: _zod.z.string().optional(),
  severity: PayloadSeveritySchema.optional(),
  timestamp: _zod.z.string().optional(),
  component: _zod.z.string().optional(),
  group: _zod.z.string().optional(),
  class: _zod.z.string().optional(),
  links: LinksSchema.optional(),
  customDetails: customDetailsSchema.optional()
}).strict().superRefine((paramsObject, ctx) => {
  const {
    timestamp,
    eventAction,
    dedupKey
  } = paramsObject;
  const convertedTimestamp = (0, _utils.convertTimestamp)(timestamp);
  if (convertedTimestamp != null) {
    try {
      const date = (0, _moment.default)(convertedTimestamp);
      if (!date.isValid()) {
        ctx.addIssue({
          code: _zod.z.ZodIssueCode.invalid_date,
          message: _i18n.i18n.translate('xpack.stackConnectors.pagerduty.invalidTimestampErrorMessage', {
            defaultMessage: `error parsing timestamp "{timestamp}"`,
            values: {
              timestamp
            }
          })
        });
        return;
      }
    } catch (err) {
      ctx.addIssue({
        code: _zod.z.ZodIssueCode.invalid_date,
        message: _i18n.i18n.translate('xpack.stackConnectors.pagerduty.timestampParsingFailedErrorMessage', {
          defaultMessage: `error parsing timestamp "{timestamp}": {message}`,
          values: {
            timestamp,
            message: err.message
          }
        })
      });
    }
  }
  if (eventAction && _constants.EVENT_ACTIONS_WITH_REQUIRED_DEDUPKEY.has(eventAction) && !dedupKey) {
    ctx.addIssue({
      code: _zod.z.ZodIssueCode.custom,
      message: _i18n.i18n.translate('xpack.stackConnectors.pagerduty.missingDedupkeyErrorMessage', {
        defaultMessage: `DedupKey is required when eventAction is "{eventAction}"`,
        values: {
          eventAction
        }
      })
    });
  }
});