"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ShodanConnector = void 0;
var _v = require("@kbn/zod/v4");
var _i18n = require("@kbn/i18n");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * Example: Shodan Connector
 *
 * This demonstrates an asset discovery connector with:
 * - Host and service searching
 * - Detailed host information
 * - Result counting
 * - Service enumeration
 *
 * MVP implementation focusing on core asset discovery actions.
 */

const ShodanConnector = exports.ShodanConnector = {
  metadata: {
    id: '.shodan',
    displayName: 'Shodan',
    description: _i18n.i18n.translate('connectorSpecs.shodan.metadata.description', {
      defaultMessage: 'Internet-wide asset discovery and vulnerability scanning'
    }),
    minimumLicense: 'gold',
    supportedFeatureIds: ['workflows']
  },
  auth: {
    types: [{
      type: 'api_key_header',
      defaults: {
        headerField: 'X-Api-Key'
      }
    }]
  },
  actions: {
    searchHosts: {
      isTool: true,
      input: _v.z.object({
        query: _v.z.string().describe('Search query'),
        page: _v.z.number().int().min(1).optional().default(1).describe('Page number')
      }),
      handler: async (ctx, input) => {
        var _ctx$secrets;
        const typedInput = input;
        const apiKey = ((_ctx$secrets = ctx.secrets) === null || _ctx$secrets === void 0 ? void 0 : _ctx$secrets.authType) === 'api_key_header' ? ctx.secrets['X-Api-Key'] : '';
        const response = await ctx.client.get('https://api.shodan.io/shodan/host/search', {
          params: {
            query: typedInput.query,
            page: typedInput.page || 1,
            key: apiKey
          }
        });
        return {
          matches: response.data.matches,
          total: response.data.total,
          facets: response.data.facets
        };
      }
    },
    getHostInfo: {
      isTool: true,
      input: _v.z.object({
        ip: _v.z.ipv4().describe('IP address')
      }),
      handler: async (ctx, input) => {
        var _ctx$secrets2;
        const typedInput = input;
        const apiKey = ((_ctx$secrets2 = ctx.secrets) === null || _ctx$secrets2 === void 0 ? void 0 : _ctx$secrets2.authType) === 'api_key_header' ? ctx.secrets['X-Api-Key'] : '';
        const response = await ctx.client.get(`https://api.shodan.io/shodan/host/${typedInput.ip}`, {
          params: {
            key: apiKey
          }
        });
        return {
          ip: response.data.ip_str,
          ports: response.data.ports,
          hostnames: response.data.hostnames,
          city: response.data.city,
          country: response.data.country_name,
          org: response.data.org,
          data: response.data.data
        };
      }
    },
    countResults: {
      isTool: true,
      input: _v.z.object({
        query: _v.z.string().describe('Search query'),
        facets: _v.z.string().optional().describe('Facets to include')
      }),
      handler: async (ctx, input) => {
        var _ctx$secrets3;
        const typedInput = input;
        const apiKey = ((_ctx$secrets3 = ctx.secrets) === null || _ctx$secrets3 === void 0 ? void 0 : _ctx$secrets3.authType) === 'api_key_header' ? ctx.secrets['X-Api-Key'] : '';
        const response = await ctx.client.get('https://api.shodan.io/shodan/host/count', {
          params: {
            query: typedInput.query,
            ...(typedInput.facets && {
              facets: typedInput.facets
            }),
            key: apiKey
          }
        });
        return {
          total: response.data.total,
          facets: response.data.facets
        };
      }
    },
    getServices: {
      isTool: true,
      input: _v.z.object({}),
      handler: async ctx => {
        var _ctx$secrets4;
        const apiKey = ((_ctx$secrets4 = ctx.secrets) === null || _ctx$secrets4 === void 0 ? void 0 : _ctx$secrets4.authType) === 'api_key_header' ? ctx.secrets['X-Api-Key'] : '';
        const response = await ctx.client.get('https://api.shodan.io/shodan/services', {
          params: {
            key: apiKey
          }
        });
        return {
          services: response.data
        };
      }
    }
  },
  test: {
    handler: async ctx => {
      try {
        var _ctx$secrets5;
        const apiKey = ((_ctx$secrets5 = ctx.secrets) === null || _ctx$secrets5 === void 0 ? void 0 : _ctx$secrets5.authType) === 'api_key_header' ? ctx.secrets['X-Api-Key'] : '';
        await ctx.client.get('https://api.shodan.io/shodan/host/8.8.8.8', {
          params: {
            key: apiKey
          }
        });
        return {
          ok: true,
          message: 'Successfully connected to Shodan API'
        };
      } catch (error) {
        return {
          ok: false,
          message: `Failed to connect: ${error}`
        };
      }
    },
    description: _i18n.i18n.translate('connectorSpecs.shodan.test.description', {
      defaultMessage: 'Verifies Shodan API key'
    })
  }
};