"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.constructUrl = constructUrl;
exports.getContentType = getContentType;
exports.getVersion = getVersion;
exports.send = send;
var _lodash = require("lodash");
var _constants = require("../../../common/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const esVersion = [];
function getVersion() {
  return esVersion;
}
function getContentType(body) {
  if (!body) return;
  return 'application/json';
}
async function send({
  http,
  method,
  path,
  data,
  asSystemRequest = false,
  withProductOrigin = false,
  asResponse = false,
  host,
  isPackagedEnvironment
}) {
  const kibanaRequestUrl = getKibanaRequestUrl(path);
  if (kibanaRequestUrl) {
    const httpMethod = method.toLowerCase();
    const url = new URL(kibanaRequestUrl);
    const {
      pathname,
      searchParams
    } = url;
    const query = {
      ...Object.fromEntries(searchParams.entries()),
      ...(isPackagedEnvironment && {
        isKibanaRequest: 'true'
      })
    };
    const body = ['post', 'put', 'patch'].includes(httpMethod) ? data : null;
    return await http[httpMethod](pathname, {
      body,
      query,
      asResponse,
      asSystemRequest
    });
  }
  return await http.post(`${_constants.API_BASE_PATH}/proxy`, {
    query: {
      path,
      method,
      ...(withProductOrigin && {
        withProductOrigin
      }),
      ...(host && {
        host
      })
    },
    body: data,
    asResponse,
    asSystemRequest
  });
}
function getKibanaRequestUrl(path) {
  const isKibanaApiRequest = path.startsWith(_constants.KIBANA_API_PREFIX);
  const kibanaBasePath = window.location.origin;
  if (isKibanaApiRequest) {
    // window.location.origin is used as a Kibana public base path for sending requests in cURL commands. E.g. "Copy as cURL".
    return `${kibanaBasePath}/${(0, _lodash.trimStart)(path.replace(_constants.KIBANA_API_PREFIX, ''), '/')}`;
  }
}
function constructUrl(baseUri, path) {
  const kibanaRequestUrl = getKibanaRequestUrl(path);
  let url = `${(0, _lodash.trimEnd)(baseUri, '/')}/${(0, _lodash.trimStart)(path, '/')}`;
  if (kibanaRequestUrl) {
    url = kibanaRequestUrl;
  }
  const {
    origin,
    pathname,
    search
  } = new URL(url);
  return `${origin}${encodePathname(pathname)}${search !== null && search !== void 0 ? search : ''}`;
}
const encodePathname = path => {
  var _URLSearchParams$get;
  const decodedPath = (_URLSearchParams$get = new URLSearchParams(`path=${path}`).get('path')) !== null && _URLSearchParams$get !== void 0 ? _URLSearchParams$get : '';

  // Skip if it is valid
  if (path === decodedPath) {
    return path;
  }
  return `/${encodeURIComponent((0, _lodash.trimStart)(decodedPath, '/'))}`;
};