"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.isCustomValidation = isCustomValidation;
exports.prepareVersionedRouteValidation = prepareVersionedRouteValidation;
exports.unwrapVersionedResponseBodyValidation = unwrapVersionedResponseBodyValidation;
var _lodash = require("lodash");
var _security_route_config_validator = require("../security_route_config_validator");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

function isCustomValidation(v) {
  return 'custom' in v;
}

/**
 * Utility for unwrapping versioned router response validation to
 * {@link RouteValidationSpec}.
 *
 * @param validation - versioned response body validation
 * @internal
 */
function unwrapVersionedResponseBodyValidation(validation) {
  if (isCustomValidation(validation)) {
    return validation.custom;
  }
  return validation();
}
function prepareValidation(validation) {
  if (validation.response) {
    const {
      unsafe,
      ...responseValidations
    } = validation.response;
    const result = {};
    for (const [key, value] of Object.entries(responseValidations)) {
      result[key] = {
        ...value
      };
      if (value.body) {
        result[key].body = isCustomValidation(value.body) ? value.body : (0, _lodash.once)(value.body);
      }
    }
    return {
      ...validation,
      response: {
        ...validation.response,
        ...result
      }
    };
  }
  return validation;
}

// Integration tested in ./core_versioned_route.test.ts
function prepareVersionedRouteValidation(options) {
  const {
    validate: originalValidate,
    security,
    ...rest
  } = options;
  let validate = originalValidate;
  if (typeof originalValidate === 'function') {
    validate = (0, _lodash.once)(() => prepareValidation(originalValidate()));
  } else if (typeof validate === 'object' && validate !== null) {
    validate = prepareValidation(validate);
  }
  return {
    security: (0, _security_route_config_validator.validRouteSecurity)(security),
    validate,
    ...rest
  };
}