"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.performBulkUpdate = void 0;
var _coreElasticsearchServerInternal = require("@kbn/core-elasticsearch-server-internal");
var _coreSavedObjectsServer = require("@kbn/core-saved-objects-server");
var _coreSavedObjectsUtilsServer = require("@kbn/core-saved-objects-utils-server");
var _coreSavedObjectsBaseServerInternal = require("@kbn/core-saved-objects-base-server-internal");
var _coreSavedObjectsApiServer = require("@kbn/core-saved-objects-api-server");
var _constants = require("../constants");
var _utils = require("./utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const performBulkUpdate = async ({
  objects,
  options
}, {
  registry,
  helpers,
  allowedTypes,
  client,
  serializer,
  extensions = {}
}) => {
  const {
    common: commonHelper,
    encryption: encryptionHelper,
    migration: migrationHelper,
    user: userHelper
  } = helpers;
  const {
    securityExtension
  } = extensions;
  const {
    migrationVersionCompatibility
  } = options;
  const namespace = commonHelper.getCurrentNamespace(options.namespace);
  const updatedBy = userHelper.getCurrentUserProfileUid();
  const time = (0, _utils.getCurrentTime)();
  let bulkGetRequestIndexCounter = 0;
  const expectedBulkGetResults = objects.map(object => {
    const {
      type,
      id,
      attributes,
      references,
      version,
      namespace: objectNamespace,
      mergeAttributes = true
    } = object;
    let error;
    if (!allowedTypes.includes(type)) {
      error = _coreSavedObjectsServer.SavedObjectsErrorHelpers.createGenericNotFoundError(type, id);
    } else {
      try {
        if (objectNamespace === _coreSavedObjectsUtilsServer.ALL_NAMESPACES_STRING) {
          error = _coreSavedObjectsServer.SavedObjectsErrorHelpers.createBadRequestError('"namespace" cannot be "*"');
        }
      } catch (e) {
        error = e;
      }
    }
    if (error) {
      return (0, _coreSavedObjectsApiServer.left)({
        id,
        type,
        error: (0, _coreSavedObjectsServer.errorContent)(error)
      });
    }
    const documentToSave = {
      [type]: attributes,
      updated_at: time,
      updated_by: updatedBy,
      ...(Array.isArray(references) && {
        references
      })
    };
    return (0, _coreSavedObjectsApiServer.right)({
      type,
      id,
      version,
      documentToSave,
      objectNamespace,
      esRequestIndex: bulkGetRequestIndexCounter++,
      migrationVersionCompatibility,
      mergeAttributes
    });
  });
  const validObjects = expectedBulkGetResults.filter(_coreSavedObjectsApiServer.isRight);
  if (validObjects.length === 0) {
    return {
      // Technically the returned array should only contain SavedObject results, but for errors this is not true (we cast to 'any' below)
      saved_objects: expectedBulkGetResults.map(({
        value
      }) => value)
    };
  }

  /*
   * Gets the namespace ID for a given object defaulting to the current namespace.
   *
   * If the `objectNamespace` is defined, it will convert it to an ID using
   * `SavedObjectsUtils.namespaceStringToId()`. Otherwise, it will use the current
   * namespace ID.
   */
  const getNamespaceId = objectNamespace => objectNamespace !== undefined ? _coreSavedObjectsUtilsServer.SavedObjectsUtils.namespaceStringToId(objectNamespace) : namespace;
  const bulkGetDocs = validObjects.map(({
    value: {
      type,
      id,
      objectNamespace
    }
  }) => ({
    _id: serializer.generateRawId(getNamespaceId(objectNamespace), type, id),
    _index: commonHelper.getIndexForType(type),
    _source: true
  }));
  const bulkGetResponse = bulkGetDocs.length ? await client.mget({
    docs: bulkGetDocs
  }, {
    ignore: [404],
    meta: true
  }) : undefined;
  // fail fast if we can't verify a 404 response is from Elasticsearch
  if (bulkGetResponse && (0, _coreElasticsearchServerInternal.isNotFoundFromUnsupportedServer)({
    statusCode: bulkGetResponse.statusCode,
    headers: bulkGetResponse.headers
  })) {
    throw _coreSavedObjectsServer.SavedObjectsErrorHelpers.createGenericNotFoundEsUnavailableError();
  }
  const authObjects = validObjects.map(element => {
    var _preflightResult$_sou;
    const {
      type,
      id,
      objectNamespace,
      esRequestIndex: index,
      documentToSave
    } = element.value;
    const preflightResult = bulkGetResponse.body.docs[index];
    const name = _coreSavedObjectsUtilsServer.SavedObjectsUtils.getName(registry.getNameAttribute(type), {
      attributes: documentToSave[type]
    });
    const accessControl = registry.supportsAccessControl(type) && securityExtension ? // @ts-expect-error MultiGetHit._source is optional
    (_preflightResult$_sou = preflightResult._source) === null || _preflightResult$_sou === void 0 ? void 0 : _preflightResult$_sou.accessControl : undefined;
    if (registry.isMultiNamespace(type)) {
      var _preflightResult$_sou2, _preflightResult$_sou3;
      return {
        type,
        id,
        objectNamespace,
        name,
        ...(accessControl && {
          accessControl
        }),
        // @ts-expect-error MultiGetHit._source is optional
        existingNamespaces: (_preflightResult$_sou2 = (_preflightResult$_sou3 = preflightResult._source) === null || _preflightResult$_sou3 === void 0 ? void 0 : _preflightResult$_sou3.namespaces) !== null && _preflightResult$_sou2 !== void 0 ? _preflightResult$_sou2 : []
      };
    } else {
      return {
        type,
        id,
        objectNamespace,
        name,
        ...(accessControl && {
          accessControl
        }),
        existingNamespaces: []
      };
    }
  });
  const authorizationResult = await (securityExtension === null || securityExtension === void 0 ? void 0 : securityExtension.authorizeBulkUpdate({
    namespace,
    objects: authObjects
  }));
  const inaccessibleObjects = authorizationResult !== null && authorizationResult !== void 0 && authorizationResult.inaccessibleObjects ? Array.from(authorizationResult.inaccessibleObjects) : [];
  const expectedAuthorizedResults = await (securityExtension === null || securityExtension === void 0 ? void 0 : securityExtension.filterInaccessibleObjectsForBulkAction(expectedBulkGetResults, inaccessibleObjects, 'bulk_update'));
  let bulkUpdateRequestIndexCounter = 0;
  const bulkUpdateParams = [];
  const expectedBulkUpdateResults = await Promise.all((expectedAuthorizedResults !== null && expectedAuthorizedResults !== void 0 ? expectedAuthorizedResults : expectedBulkGetResults).map(async expectedBulkGetResult => {
    var _savedObjectNamespace;
    if ((0, _coreSavedObjectsApiServer.isLeft)(expectedBulkGetResult)) {
      return expectedBulkGetResult;
    }
    const {
      esRequestIndex,
      id,
      type,
      version,
      documentToSave,
      objectNamespace,
      mergeAttributes
    } = expectedBulkGetResult.value;
    const versionProperties = (0, _utils.getExpectedVersionProperties)(version);
    const indexFound = (bulkGetResponse === null || bulkGetResponse === void 0 ? void 0 : bulkGetResponse.statusCode) !== 404;
    const actualResult = indexFound ? bulkGetResponse === null || bulkGetResponse === void 0 ? void 0 : bulkGetResponse.body.docs[esRequestIndex] : undefined;
    const docFound = indexFound && (0, _utils.isMgetDoc)(actualResult) && actualResult.found;
    const isMultiNS = registry.isMultiNamespace(type);
    if (!docFound || isMultiNS && !(0, _utils.rawDocExistsInNamespace)(registry, actualResult, getNamespaceId(objectNamespace))) {
      return (0, _coreSavedObjectsApiServer.left)({
        id,
        type,
        error: (0, _coreSavedObjectsServer.errorContent)(_coreSavedObjectsServer.SavedObjectsErrorHelpers.createGenericNotFoundError(type, id))
      });
    }
    let savedObjectNamespace;
    let savedObjectNamespaces;
    if (isMultiNS) {
      var _source$namespaces;
      // @ts-expect-error MultiGetHit is incorrectly missing _id, _source
      savedObjectNamespaces = (_source$namespaces = actualResult._source.namespaces) !== null && _source$namespaces !== void 0 ? _source$namespaces : [
      // @ts-expect-error MultiGetHit is incorrectly missing _id, _source
      _coreSavedObjectsUtilsServer.SavedObjectsUtils.namespaceIdToString(actualResult._source.namespace)];
    } else if (registry.isSingleNamespace(type)) {
      // if `objectNamespace` is undefined, fall back to `options.namespace`
      savedObjectNamespace = getNamespaceId(objectNamespace);
    }
    const document = (0, _utils.getSavedObjectFromSource)(registry, type, id, actualResult, {
      migrationVersionCompatibility
    });
    let migrated;
    try {
      migrated = migrationHelper.migrateStorageDocument(document);
    } catch (migrateStorageDocError) {
      throw _coreSavedObjectsServer.SavedObjectsErrorHelpers.decorateGeneralError(migrateStorageDocError, 'Failed to migrate document to the latest version.');
    }
    const typeDefinition = registry.getType(type);
    const encryptedUpdatedAttributes = await encryptionHelper.optionallyEncryptAttributes(type, id, objectNamespace || namespace, documentToSave[type]);
    const updatedAttributes = mergeAttributes ? (0, _utils.mergeForUpdate)({
      targetAttributes: {
        ...migrated.attributes
      },
      updatedAttributes: encryptedUpdatedAttributes,
      typeMappings: typeDefinition.mappings
    }) : encryptedUpdatedAttributes;
    const migratedUpdatedSavedObjectDoc = migrationHelper.migrateInputDocument({
      ...migrated,
      id,
      type,
      ...(savedObjectNamespace && {
        namespace: savedObjectNamespace
      }),
      ...(savedObjectNamespaces && {
        namespaces: savedObjectNamespaces
      }),
      attributes: updatedAttributes,
      updated_at: time,
      updated_by: updatedBy,
      ...(migrated.accessControl ? {
        accessControl: migrated.accessControl
      } : {}),
      ...(Array.isArray(documentToSave.references) && {
        references: documentToSave.references
      })
    });
    const updatedMigratedDocumentToSave = serializer.savedObjectToRaw(migratedUpdatedSavedObjectDoc);
    const namespaces = (_savedObjectNamespace = savedObjectNamespaces) !== null && _savedObjectNamespace !== void 0 ? _savedObjectNamespace : savedObjectNamespace ? [savedObjectNamespace] : [];
    const expectedResult = {
      type,
      id,
      namespaces,
      esRequestIndex: bulkUpdateRequestIndexCounter++,
      documentToSave: expectedBulkGetResult.value.documentToSave,
      rawMigratedUpdatedDoc: updatedMigratedDocumentToSave,
      migrationVersionCompatibility
    };
    bulkUpdateParams.push({
      index: {
        _id: serializer.generateRawId(getNamespaceId(objectNamespace), type, id),
        _index: commonHelper.getIndexForType(type),
        ...versionProperties
      }
    }, updatedMigratedDocumentToSave._source);
    return (0, _coreSavedObjectsApiServer.right)(expectedResult);
  }));
  const {
    refresh = _constants.DEFAULT_REFRESH_SETTING
  } = options;
  const bulkUpdateResponse = bulkUpdateParams.length ? await client.bulk({
    refresh,
    operations: bulkUpdateParams,
    _source_includes: ['originId'],
    require_alias: true
  }) : undefined;
  const result = {
    saved_objects: expectedBulkUpdateResults.map(expectedResult => {
      var _bulkUpdateResponse$i;
      if ((0, _coreSavedObjectsApiServer.isLeft)(expectedResult)) {
        return expectedResult.value;
      }
      const {
        type,
        id,
        documentToSave,
        esRequestIndex,
        rawMigratedUpdatedDoc
      } = expectedResult.value;
      const response = (_bulkUpdateResponse$i = bulkUpdateResponse === null || bulkUpdateResponse === void 0 ? void 0 : bulkUpdateResponse.items[esRequestIndex]) !== null && _bulkUpdateResponse$i !== void 0 ? _bulkUpdateResponse$i : {};
      const rawResponse = Object.values(response)[0];
      const error = (0, _utils.getBulkOperationError)(type, id, rawResponse);
      if (error) {
        return {
          type,
          id,
          error
        };
      }
      const {
        _seq_no: seqNo,
        _primary_term: primaryTerm
      } = rawResponse;
      const {
        [type]: attributes,
        references,
        updated_at,
        updated_by
      } = documentToSave;
      const {
        originId,
        namespaces: docNamespaces,
        namespace: docNamespace
      } = rawMigratedUpdatedDoc._source;
      return {
        id,
        type,
        ...(registry.isMultiNamespace(type) && {
          namespaces: docNamespaces
        }),
        ...(registry.isSingleNamespace(type) && {
          namespaces: [_coreSavedObjectsUtilsServer.SavedObjectsUtils.namespaceIdToString(docNamespace)]
        }),
        ...(originId && {
          originId
        }),
        updated_at,
        updated_by,
        ...(registry.supportsAccessControl(type) && {
          accessControl: rawMigratedUpdatedDoc._source.accessControl
        }),
        version: (0, _coreSavedObjectsBaseServerInternal.encodeVersion)(seqNo, primaryTerm),
        attributes,
        references
      };
    })
  };
  return encryptionHelper.optionallyDecryptAndRedactBulkResult(result, authorizationResult === null || authorizationResult === void 0 ? void 0 : authorizationResult.typeMap, objects);
};
exports.performBulkUpdate = performBulkUpdate;