"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.performDelete = void 0;
var _coreElasticsearchServerInternal = require("@kbn/core-elasticsearch-server-internal");
var _coreSavedObjectsServer = require("@kbn/core-saved-objects-server");
var _coreSavedObjectsUtilsServer = require("@kbn/core-saved-objects-utils-server");
var _constants = require("../constants");
var _delete_legacy_url_aliases = require("./internals/delete_legacy_url_aliases");
var _utils = require("./utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const performDelete = async ({
  type,
  id,
  options
}, {
  registry,
  helpers,
  allowedTypes,
  client,
  serializer,
  extensions = {},
  logger,
  mappings
}) => {
  const {
    common: commonHelper,
    preflight: preflightHelper
  } = helpers;
  const {
    securityExtension
  } = extensions;
  const namespace = commonHelper.getCurrentNamespace(options.namespace);
  if (!allowedTypes.includes(type)) {
    throw _coreSavedObjectsServer.SavedObjectsErrorHelpers.createGenericNotFoundError(type, id);
  }
  const {
    refresh = _constants.DEFAULT_REFRESH_SETTING,
    force
  } = options;
  if (securityExtension) {
    var _savedObjectResponse$, _savedObjectResponse$2;
    const nameAttribute = registry.getNameAttribute(type);
    const savedObjectResponse = await client.get({
      index: commonHelper.getIndexForType(type),
      id: serializer.generateRawId(namespace, type, id),
      _source_includes: [..._coreSavedObjectsUtilsServer.SavedObjectsUtils.getIncludedNameFields(type, nameAttribute), 'accessControl']
    }, {
      ignore: [404],
      meta: true
    });
    const saveObject = {
      attributes: (_savedObjectResponse$ = savedObjectResponse.body._source) === null || _savedObjectResponse$ === void 0 ? void 0 : _savedObjectResponse$[type]
    };
    const name = securityExtension.includeSavedObjectNames() ? _coreSavedObjectsUtilsServer.SavedObjectsUtils.getName(nameAttribute, saveObject) : undefined;
    const accessControl = (_savedObjectResponse$2 = savedObjectResponse.body._source) === null || _savedObjectResponse$2 === void 0 ? void 0 : _savedObjectResponse$2.accessControl;
    // we don't need to pass existing namespaces in because we're only concerned with authorizing
    // the current space. This saves us from performing the preflight check if we're unauthorized
    await (securityExtension === null || securityExtension === void 0 ? void 0 : securityExtension.authorizeDelete({
      namespace,
      object: {
        type,
        id,
        name,
        accessControl
      }
    }));
  }
  const rawId = serializer.generateRawId(namespace, type, id);
  let preflightResult;
  if (registry.isMultiNamespace(type)) {
    var _preflightResult$save;
    // note: this check throws an error if the object is found but does not exist in this namespace
    preflightResult = await preflightHelper.preflightCheckNamespaces({
      type,
      id,
      namespace
    });
    if (preflightResult.checkResult === 'found_outside_namespace' || preflightResult.checkResult === 'not_found') {
      throw _coreSavedObjectsServer.SavedObjectsErrorHelpers.createGenericNotFoundError(type, id);
    }
    const existingNamespaces = (_preflightResult$save = preflightResult.savedObjectNamespaces) !== null && _preflightResult$save !== void 0 ? _preflightResult$save : [];
    if (!force && (existingNamespaces.length > 1 || existingNamespaces.includes(_coreSavedObjectsUtilsServer.ALL_NAMESPACES_STRING))) {
      throw _coreSavedObjectsServer.SavedObjectsErrorHelpers.createBadRequestError('Unable to delete saved object that exists in multiple namespaces, use the `force` option to delete it anyway');
    }
  }
  const {
    body,
    statusCode,
    headers
  } = await client.delete({
    id: rawId,
    index: commonHelper.getIndexForType(type),
    ...(0, _utils.getExpectedVersionProperties)(undefined),
    refresh
  }, {
    ignore: [404],
    meta: true
  });
  if ((0, _coreElasticsearchServerInternal.isNotFoundFromUnsupportedServer)({
    statusCode,
    headers
  })) {
    throw _coreSavedObjectsServer.SavedObjectsErrorHelpers.createGenericNotFoundEsUnavailableError(type, id);
  }
  const deleted = body.result === 'deleted';
  if (deleted) {
    var _preflightResult;
    const namespaces = (_preflightResult = preflightResult) === null || _preflightResult === void 0 ? void 0 : _preflightResult.savedObjectNamespaces;
    if (namespaces) {
      // This is a multi-namespace object type, and it might have legacy URL aliases that need to be deleted.
      await (0, _delete_legacy_url_aliases.deleteLegacyUrlAliases)({
        mappings,
        registry,
        client,
        getIndexForType: commonHelper.getIndexForType.bind(commonHelper),
        type,
        id,
        ...(namespaces.includes(_coreSavedObjectsUtilsServer.ALL_NAMESPACES_STRING) ? {
          namespaces: [],
          deleteBehavior: 'exclusive'
        } // delete legacy URL aliases for this type/ID for all spaces
        : {
          namespaces,
          deleteBehavior: 'inclusive'
        }) // delete legacy URL aliases for this type/ID for these specific spaces
      }).catch(err => {
        // The object has already been deleted, but we caught an error when attempting to delete aliases.
        // A consumer cannot attempt to delete the object again, so just log the error and swallow it.
        logger.error(`Unable to delete aliases when deleting an object: ${err.message}`);
      });
    }
    return {};
  }
  const deleteDocNotFound = body.result === 'not_found';
  // @ts-expect-error @elastic/elasticsearch doesn't declare error on DeleteResponse
  const deleteIndexNotFound = body.error && body.error.type === 'index_not_found_exception';
  if (deleteDocNotFound || deleteIndexNotFound) {
    // see "404s from missing index" above
    throw _coreSavedObjectsServer.SavedObjectsErrorHelpers.createGenericNotFoundError(type, id);
  }
  throw new Error(`Unexpected Elasticsearch DELETE response: ${JSON.stringify({
    type,
    id,
    response: {
      body,
      statusCode
    }
  })}`);
};
exports.performDelete = performDelete;