"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.isSavedObjectsChangeOwnershipOptions = exports.isSavedObjectsChangeAccessModeOptions = exports.changeObjectAccessControl = void 0;
var _coreElasticsearchServerInternal = require("@kbn/core-elasticsearch-server-internal");
var _coreSavedObjectsServer = require("@kbn/core-saved-objects-server");
var _coreSavedObjectsApiServer = require("@kbn/core-saved-objects-api-server");
var _utils = require("../utils");
var _internal_utils = require("../utils/internal_utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

// Regular expression to validate user profile IDs as generated by Elasticsearch
// User profile IDs are expected to be in the format "u_<principal>_<version>"
const USER_PROFILE_REGEX = /^u_.+_.+$/;
const isValidUserProfileId = userProfileId => {
  return USER_PROFILE_REGEX.test(userProfileId);
};
const isSavedObjectsChangeAccessModeOptions = options => {
  return 'accessMode' in options;
};
exports.isSavedObjectsChangeAccessModeOptions = isSavedObjectsChangeAccessModeOptions;
const isSavedObjectsChangeOwnershipOptions = options => {
  return 'newOwnerProfileUid' in options;
};
exports.isSavedObjectsChangeOwnershipOptions = isSavedObjectsChangeOwnershipOptions;
const VALID_ACCESS_MODES = ['default', 'write_restricted'];
const validateChangeAccessControlParams = ({
  actionType,
  newOwnerProfileUid,
  accessMode,
  objects
}) => {
  if (actionType === 'changeOwnership') {
    if (!newOwnerProfileUid) {
      throw _coreSavedObjectsServer.SavedObjectsErrorHelpers.createBadRequestError('The "newOwnerProfileUid" field is required to change ownership of a saved object.');
    }
    if (!isValidUserProfileId(newOwnerProfileUid)) {
      throw _coreSavedObjectsServer.SavedObjectsErrorHelpers.createBadRequestError(`User profile ID is invalid: expected "u_<principal>_<version>"`);
    }
  }
  if (actionType === 'changeAccessMode' && accessMode === undefined) {
    throw _coreSavedObjectsServer.SavedObjectsErrorHelpers.createBadRequestError('The "accessMode" field is required to change access mode of a saved object.');
  }
  if (actionType === 'changeAccessMode' && accessMode !== undefined && !VALID_ACCESS_MODES.includes(accessMode)) {
    throw _coreSavedObjectsServer.SavedObjectsErrorHelpers.createBadRequestError('When specified, the "accessMode" field can only be "default" or "write_restricted".');
  }
  if (!objects || objects.length === 0) {
    throw _coreSavedObjectsServer.SavedObjectsErrorHelpers.createBadRequestError(`No objects specified for ${actionType === 'changeOwnership' ? 'ownership change' : 'access mode change'}`);
  }
};
const changeObjectAccessControl = async params => {
  const {
    namespace
  } = params.options;
  const {
    actionType,
    currentUserProfileUid
  } = params;

  // Extract owner for changeOwnership or accessMode for changeAccessMode
  const newOwnerProfileUid = actionType === 'changeOwnership' && isSavedObjectsChangeOwnershipOptions(params.options) ? params.options.newOwnerProfileUid : undefined;
  const accessMode = actionType === 'changeAccessMode' && isSavedObjectsChangeAccessModeOptions(params.options) ? params.options.accessMode : undefined;
  const {
    registry,
    allowedTypes,
    client,
    serializer,
    getIndexForType,
    objects,
    securityExtension
  } = params;
  if (!securityExtension) {
    throw _coreSavedObjectsServer.SavedObjectsErrorHelpers.createBadRequestError('Unable to proceed with changing access control without security extension.');
  }
  validateChangeAccessControlParams({
    actionType,
    newOwnerProfileUid,
    accessMode,
    objects
  });

  /**
   * We create a list of expected bulk get results, which will be used to
   * perform the authorization checks and to build the bulk operation request.
   * Valid objects will then be used for rewriting back to the index once authz and access control
   * checks are done and valid.
   */

  const expectedBulkGetResults = objects.map((object, index) => {
    const {
      type,
      id
    } = object;
    if (!allowedTypes.includes(type)) {
      const error = _coreSavedObjectsServer.SavedObjectsErrorHelpers.createGenericNotFoundError(type, id);
      return (0, _coreSavedObjectsApiServer.left)({
        id,
        type,
        error
      });
    }
    if (!registry.supportsAccessControl(type)) {
      const error = _coreSavedObjectsServer.SavedObjectsErrorHelpers.createBadRequestError(`The type ${type} does not support access control`);
      return (0, _coreSavedObjectsApiServer.left)({
        id,
        type,
        error
      });
    }
    return (0, _coreSavedObjectsApiServer.right)({
      type,
      id,
      esRequestIndex: index
    });
  });
  const validObjects = expectedBulkGetResults.filter(_coreSavedObjectsApiServer.isRight);
  if (validObjects.length === 0) {
    // We only have error results; return early to avoid potentially trying authZ checks for 0 types which would result in an exception.
    return {
      objects: expectedBulkGetResults.map(({
        value
      }) => value)
    };
  }

  // Only get the objects that are required for processing
  const bulkGetDocs = validObjects.map(x => ({
    _id: serializer.generateRawId(undefined, x.value.type, x.value.id),
    _index: getIndexForType(x.value.type),
    _source: ['type', 'namespaces', 'accessControl']
  }));
  const bulkGetResponse = await client.mget({
    docs: bulkGetDocs
  }, {
    ignore: [404],
    meta: true
  });
  if (bulkGetResponse && (0, _coreElasticsearchServerInternal.isNotFoundFromUnsupportedServer)({
    statusCode: bulkGetResponse.statusCode,
    headers: bulkGetResponse.headers
  })) {
    throw _coreSavedObjectsServer.SavedObjectsErrorHelpers.createGenericNotFoundEsUnavailableError();
  }
  const authObjects = validObjects.map(element => {
    var _preflightResult$_sou, _preflightResult$_sou2, _preflightResult$_sou3;
    const {
      type,
      id,
      esRequestIndex: index
    } = element.value;
    const preflightResult = bulkGetResponse.body.docs[index];
    return {
      type,
      id,
      existingNamespaces: (_preflightResult$_sou = preflightResult === null || preflightResult === void 0 ? void 0 : (_preflightResult$_sou2 = preflightResult._source) === null || _preflightResult$_sou2 === void 0 ? void 0 : _preflightResult$_sou2.namespaces) !== null && _preflightResult$_sou !== void 0 ? _preflightResult$_sou : [],
      accessControl: preflightResult === null || preflightResult === void 0 ? void 0 : (_preflightResult$_sou3 = preflightResult._source) === null || _preflightResult$_sou3 === void 0 ? void 0 : _preflightResult$_sou3.accessControl
    };
  });
  const authorizationResult = await securityExtension.authorizeChangeAccessControl({
    namespace,
    objects: authObjects
  }, actionType);
  const time = new Date().toISOString();
  let bulkOperationRequestIndexCounter = 0;
  const bulkOperationParams = [];
  const expectedBulkOperationResults = expectedBulkGetResults.map(expectedBulkGetResult => {
    if ((0, _coreSavedObjectsApiServer.isLeft)(expectedBulkGetResult)) {
      return expectedBulkGetResult;
    }
    const {
      id,
      type,
      esRequestIndex
    } = expectedBulkGetResult.value;
    const rawDoc = bulkGetResponse.body.docs[esRequestIndex];
    if ((0, _internal_utils.isMgetError)(rawDoc) || !(rawDoc !== null && rawDoc !== void 0 && rawDoc.found)) {
      const error = _coreSavedObjectsServer.SavedObjectsErrorHelpers.createGenericNotFoundError(type, id);
      return (0, _coreSavedObjectsApiServer.left)({
        id,
        type,
        error
      });
    }
    if (!(0, _utils.rawDocExistsInNamespace)(registry, rawDoc, namespace)) {
      const error = _coreSavedObjectsServer.SavedObjectsErrorHelpers.createGenericNotFoundError(type, id);
      return (0, _coreSavedObjectsApiServer.left)({
        id,
        type,
        error
      });
    }
    if (authorizationResult.status !== 'fully_authorized' && !authorizationResult.typeMap.has(type)) {
      const error = _coreSavedObjectsServer.SavedObjectsErrorHelpers.decorateForbiddenError(new Error(`User is not authorized to ${actionType === 'changeOwnership' ? 'change ownership' : 'change access mode'} of type "${type}".`));
      return (0, _coreSavedObjectsApiServer.left)({
        id,
        type,
        error
      });
    }
    const currentSource = rawDoc._source;
    const versionProperties = (0, _utils.getExpectedVersionProperties)(undefined, rawDoc);
    const documentMetadata = {
      _id: serializer.generateRawId(undefined, type, id),
      _index: getIndexForType(type),
      ...versionProperties
    };
    let documentToSave;
    if (actionType === 'changeOwnership') {
      documentToSave = {
        updated_at: time,
        updated_by: currentUserProfileUid,
        accessControl: {
          ...((currentSource === null || currentSource === void 0 ? void 0 : currentSource.accessControl) || {
            accessMode: 'default'
          }),
          owner: newOwnerProfileUid
        }
      };
    } else {
      documentToSave = {
        updated_at: time,
        updated_by: currentUserProfileUid,
        accessControl: {
          ...((currentSource === null || currentSource === void 0 ? void 0 : currentSource.accessControl) || {
            owner: currentUserProfileUid
          }),
          accessMode: accessMode !== null && accessMode !== void 0 ? accessMode : 'default'
        }
      };
    }
    bulkOperationParams.push({
      update: documentMetadata
    }, {
      doc: documentToSave
    });
    return (0, _coreSavedObjectsApiServer.right)({
      type,
      id,
      esRequestIndex: bulkOperationRequestIndexCounter++,
      doc: rawDoc
    });
  });
  const bulkOperationResponse = bulkOperationParams.length ? await client.bulk({
    refresh: 'wait_for',
    operations: bulkOperationParams,
    require_alias: true
  }) : undefined;
  return {
    objects: expectedBulkOperationResults.map(expectedResult => {
      var _bulkOperationRespons;
      if ((0, _coreSavedObjectsApiServer.isLeft)(expectedResult)) {
        return expectedResult.value;
      }
      const {
        type,
        id,
        esRequestIndex
      } = expectedResult.value;
      const response = (_bulkOperationRespons = bulkOperationResponse === null || bulkOperationResponse === void 0 ? void 0 : bulkOperationResponse.items[esRequestIndex]) !== null && _bulkOperationRespons !== void 0 ? _bulkOperationRespons : {};
      const rawResponse = Object.values(response)[0];
      const error = (0, _utils.getBulkOperationError)(type, id, rawResponse);
      if (error) {
        return {
          id,
          type,
          error
        };
      }
      return {
        id,
        type
      };
    })
  };
};
exports.changeObjectAccessControl = changeObjectAccessControl;