"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.performRemoveReferencesTo = void 0;
var _coreElasticsearchServerInternal = require("@kbn/core-elasticsearch-server-internal");
var _coreSavedObjectsServer = require("@kbn/core-saved-objects-server");
var _coreSavedObjectsUtilsServer = require("@kbn/core-saved-objects-utils-server");
var _search = require("../search");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const performRemoveReferencesTo = async ({
  type,
  id,
  options
}, {
  registry,
  helpers,
  client,
  mappings,
  serializer,
  extensions = {}
}) => {
  var _body$failures;
  const {
    common: commonHelper
  } = helpers;
  const {
    securityExtension
  } = extensions;
  const namespace = commonHelper.getCurrentNamespace(options.namespace);
  const {
    refresh = true
  } = options;
  if (securityExtension) {
    let name;
    if (securityExtension.includeSavedObjectNames()) {
      var _savedObjectResponse$;
      const nameAttribute = registry.getNameAttribute(type);
      const savedObjectResponse = await client.get({
        index: commonHelper.getIndexForType(type),
        id: serializer.generateRawId(namespace, type, id),
        _source_includes: _coreSavedObjectsUtilsServer.SavedObjectsUtils.getIncludedNameFields(type, nameAttribute)
      }, {
        ignore: [404],
        meta: true
      });
      const saveObject = {
        attributes: (_savedObjectResponse$ = savedObjectResponse.body._source) === null || _savedObjectResponse$ === void 0 ? void 0 : _savedObjectResponse$[type]
      };
      name = _coreSavedObjectsUtilsServer.SavedObjectsUtils.getName(nameAttribute, saveObject);
    }
    await securityExtension.authorizeRemoveReferences({
      namespace,
      object: {
        type,
        id,
        name
      }
    });
  }
  const allTypes = registry.getAllTypes().map(t => t.name);

  // we need to target all SO indices as all types of objects may have references to the given SO.
  const targetIndices = commonHelper.getIndicesForTypes(allTypes);
  const {
    body,
    statusCode,
    headers
  } = await client.updateByQuery({
    index: targetIndices,
    refresh,
    script: {
      source: `
          if (ctx._source.containsKey('references')) {
            def items_to_remove = [];
            for (item in ctx._source.references) {
              if ( (item['type'] == params['type']) && (item['id'] == params['id']) ) {
                items_to_remove.add(item);
              }
            }
            ctx._source.references.removeAll(items_to_remove);
          }
        `,
      params: {
        type,
        id
      },
      lang: 'painless'
    },
    conflicts: 'proceed',
    ...(0, _search.getSearchDsl)(mappings, registry, {
      namespaces: namespace ? [namespace] : undefined,
      type: allTypes,
      hasReference: {
        type,
        id
      }
    }) // TS is complaining and it's unlikely that we sort here
  }, {
    ignore: [404],
    meta: true
  });
  // fail fast if we can't verify a 404 is from Elasticsearch
  if ((0, _coreElasticsearchServerInternal.isNotFoundFromUnsupportedServer)({
    statusCode,
    headers
  })) {
    throw _coreSavedObjectsServer.SavedObjectsErrorHelpers.createGenericNotFoundEsUnavailableError(type, id);
  }
  if ((_body$failures = body.failures) !== null && _body$failures !== void 0 && _body$failures.length) {
    throw _coreSavedObjectsServer.SavedObjectsErrorHelpers.createConflictError(type, id, `${body.failures.length} references could not be removed`);
  }
  return {
    updated: body.updated
  };
};
exports.performRemoveReferencesTo = performRemoveReferencesTo;