"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerImportRoute = void 0;
var _nodePath = _interopRequireDefault(require("node:path"));
var _path = require("path");
var _configSchema = require("@kbn/config-schema");
var _coreSavedObjectsImportExportServerInternal = require("@kbn/core-saved-objects-import-export-server-internal");
var _shared_schemas = require("./shared_schemas");
var _utils = require("./utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const registerImportRoute = (router, {
  config,
  coreUsageData
}) => {
  const {
    maxImportPayloadBytes
  } = config;
  router.post({
    path: '/_import',
    options: {
      summary: `Import saved objects`,
      tags: ['oas-tag:saved objects'],
      access: 'public',
      description: `Create sets of Kibana saved objects from a file created by the export API. Saved objects can only be imported into the same version, a newer minor on the same major, or the next major. Tampering with exported data risks introducing unspecified errors and data loss.

Exported saved objects are not backwards compatible and cannot be imported into an older version of Kibana.`,
      body: {
        maxBytes: maxImportPayloadBytes,
        output: 'stream',
        accepts: 'multipart/form-data'
      },
      oasOperationObject: () => _nodePath.default.resolve(__dirname, './import.examples.yaml')
    },
    security: {
      authz: {
        enabled: false,
        reason: 'This route delegates authorization to the Saved Objects Client'
      }
    },
    validate: {
      request: {
        query: _configSchema.schema.object({
          overwrite: _configSchema.schema.boolean({
            defaultValue: false,
            meta: {
              description: 'Overwrites saved objects when they already exist. When used, potential conflict errors are automatically resolved by overwriting the destination object. NOTE: This option cannot be used with the `createNewCopies` option.'
            }
          }),
          createNewCopies: _configSchema.schema.boolean({
            defaultValue: false,
            meta: {
              description: 'Creates copies of saved objects, regenerates each object ID, and resets the origin. When used, potential conflict errors are avoided. NOTE: This option cannot be used with the `overwrite` and `compatibilityMode` options.'
            }
          }),
          compatibilityMode: _configSchema.schema.boolean({
            defaultValue: false,
            meta: {
              description: 'Applies various adjustments to the saved objects that are being imported to maintain compatibility between different Kibana versions. Use this option only if you encounter issues with imported saved objects. NOTE: This option cannot be used with the `createNewCopies` option.'
            }
          })
        }, {
          validate: object => {
            if (object.overwrite && object.createNewCopies) {
              return 'cannot use [overwrite] with [createNewCopies]';
            }
            if (object.createNewCopies && object.compatibilityMode) {
              return 'cannot use [createNewCopies] with [compatibilityMode]';
            }
          }
        }),
        body: _configSchema.schema.object({
          file: _configSchema.schema.stream({
            meta: {
              description: 'A file exported using the export API. Changing the contents of the exported file in any way before importing it can cause errors, crashes or data loss. NOTE: The `savedObjects.maxImportExportSize` configuration setting limits the number of saved objects which may be included in this file. Similarly, the `savedObjects.maxImportPayloadBytes` setting limits the overall size of the file that can be imported.'
            }
          })
        })
      },
      response: {
        200: {
          description: 'Indicates a successful call.',
          bodyContentType: 'application/json',
          body: okResponseSchema
        },
        400: (0, _shared_schemas.badResponseSchema)()
      }
    }
  }, (0, _utils.catchAndReturnBoomErrors)(async (context, request, response) => {
    const {
      overwrite,
      createNewCopies,
      compatibilityMode
    } = request.query;
    const {
      getClient,
      getImporter,
      typeRegistry
    } = (await context.core).savedObjects;
    const usageStatsClient = coreUsageData.getClient();
    usageStatsClient.incrementSavedObjectsImport({
      request,
      createNewCopies,
      overwrite,
      compatibilityMode
    }).catch(() => {});
    const file = request.body.file;
    const fileExtension = (0, _path.extname)(file.hapi.filename).toLowerCase();
    if (fileExtension !== '.ndjson') {
      return response.badRequest({
        body: `Invalid file extension ${fileExtension}`
      });
    }
    let readStream;
    try {
      readStream = await (0, _utils.createSavedObjectsStreamFromNdJson)(file);
    } catch (e) {
      return response.badRequest({
        body: e
      });
    }
    const supportedTypes = typeRegistry.getImportableAndExportableTypes().map(t => t.name);
    const includedHiddenTypes = supportedTypes.filter(supportedType => typeRegistry.isHidden(supportedType));
    const client = getClient({
      includedHiddenTypes
    });
    const importer = getImporter(client);
    try {
      const result = await importer.import({
        readStream,
        overwrite,
        createNewCopies,
        compatibilityMode
      });
      return response.ok({
        body: result
      });
    } catch (e) {
      if (e instanceof _coreSavedObjectsImportExportServerInternal.SavedObjectsImportError) {
        return response.badRequest({
          body: {
            message: e.message,
            attributes: e.attributes
          }
        });
      }
      throw e;
    }
  }));
};
exports.registerImportRoute = registerImportRoute;
const okResponseSchema = () => _configSchema.schema.object({
  success: _configSchema.schema.boolean({
    meta: {
      description: 'Indicates when the import was successfully completed. When set to false, some objects may not have been created. For additional information, refer to the `errors` and `successResults` properties.'
    }
  }),
  successCount: _configSchema.schema.number({
    meta: {
      description: 'Indicates the number of successfully imported records.'
    }
  }),
  errors: _configSchema.schema.arrayOf(_configSchema.schema.object({}, {
    unknowns: 'allow'
  }), {
    meta: {
      description: `Indicates the import was unsuccessful and specifies the objects that failed to import.

NOTE: One object may result in multiple errors, which requires separate steps to resolve. For instance, a \`missing_references\` error and conflict error.`
    }
  }),
  successResults: _configSchema.schema.arrayOf(_configSchema.schema.object({}, {
    unknowns: 'allow'
  }), {
    meta: {
      description: `Indicates the objects that are successfully imported, with any metadata if applicable.

NOTE: Objects are created only when all resolvable errors are addressed, including conflicts and missing references. If objects are created as new copies, each entry in the \`successResults\` array includes a \`destinationId\` attribute.`
    }
  })
});