"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.parseAppUrl = void 0;
var _std = require("@kbn/std");
var _url = require("url");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * Parse given URL and return the associated app id and path if any app matches, or undefined if none do.
 * Input can either be:
 *
 * - an absolute path containing the basePath,
 *   e.g `/base-path/app/my-app/some-path`
 *
 * - an absolute URL matching the `origin` of the Kibana instance (as seen by the browser),
 *   e.g `https://kibana:8080/base-path/app/my-app/some-path`
 *
 * - a path relative to the provided `currentUrl`.
 *   e.g with `currentUrl` being `https://kibana:8080/base-path/app/current-app/some-path`
 *   `../other-app/other-path` will be converted to `/base-path/app/other-app/other-path`
 */
const parseAppUrl = (url, basePath, apps, currentUrl = window.location.href) => {
  const currentOrigin = (0, _std.getUrlOrigin)(currentUrl);
  if (!currentOrigin) {
    throw new Error('when manually provided, currentUrl must be valid url with an origin');
  }
  const currentPath = currentUrl.substring(currentOrigin.length);

  // remove the origin from the given url
  if (url.startsWith(currentOrigin)) {
    url = url.substring(currentOrigin.length);
  }

  // if the path is relative (i.e `../../to/somewhere`), we convert it to absolute
  if (!url.startsWith('/')) {
    url = (0, _url.resolve)(currentPath, url);
  }

  // if using a basePath and the absolute path does not starts with it, it can't be a match
  const basePathValue = basePath.get();
  if (basePathValue && !url.startsWith(basePathValue)) {
    return undefined;
  }
  url = basePath.remove(url);
  if (!url.startsWith('/')) {
    return undefined;
  }
  for (const app of apps.values()) {
    const appPath = app.appRoute || `/app/${app.id}`;
    if (urlInApp(url, appPath)) {
      const path = url.substr(appPath.length);
      return {
        app: app.id,
        path: path.length ? path : undefined
      };
    }
  }
};
exports.parseAppUrl = parseAppUrl;
const separators = ['/', '?', '#'];
const urlInApp = (url, appPath) => {
  if (url === appPath) {
    return true;
  }
  if (url.startsWith(appPath)) {
    const nextChar = url.substring(appPath.length, appPath.length + 1);
    return separators.includes(nextChar);
  }
  return false;
};