"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.defaultSrcChildDirectives = exports.defaultRules = exports.defaultReportOnlyRules = exports.additionalRules = exports.CspDirectives = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _deepmerge = _interopRequireDefault(require("deepmerge"));
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * The default report only directives rules
 */
const defaultReportOnlyRules = exports.defaultReportOnlyRules = {
  'form-action': [`'report-sample'`, `'self'`],
  'default-src': [`'report-sample'`, `'none'`],
  'font-src': [`'report-sample'`, `'self'`],
  'img-src': [`'report-sample'`, `'self'`, 'data:', 'tiles.maps.elastic.co'],
  // Same as below for tiles.maps.elastic.co
  'connect-src': [`'report-sample'`, `'self'`,
  // TODO: Ideally, Core would not know about these endpoints, as they are governed by the Telemetry plugin.
  // This can be improved once https://github.com/elastic/kibana/issues/181812 is implemented.
  'telemetry.elastic.co', 'telemetry-staging.elastic.co',
  // Same as above, but these endpoints are related to maps
  'feeds.elastic.co', 'tiles.maps.elastic.co', 'vector.maps.elastic.co']
};

/**
 * The default directives rules that are always applied
 */
const defaultRules = exports.defaultRules = {
  'script-src': [`'report-sample'`, `'self'`],
  'worker-src': [`'report-sample'`, `'self'`, `blob:`],
  'style-src': [`'report-sample'`, `'self'`, `'unsafe-inline'`],
  'object-src': [`'report-sample'`, `'none'`]
};

/**
 * Per-directive rules that will be added when the configuration contains at least one value
 * Main purpose is to add `self` value to some directives when the configuration specifies other values
 */
const additionalRules = exports.additionalRules = {
  'connect-src': [`'self'`],
  'default-src': [`'self'`],
  'font-src': [`'self'`],
  'img-src': [`'self'`],
  'frame-ancestors': [`'self'`],
  'frame-src': [`'self'`]
};

/**
 * Child directives that should inherit from `default-src` if not explicitly set.
 * Reference: https://developer.mozilla.org/en-US/docs/Web/HTTP/Reference/Headers/Content-Security-Policy/default-src
 */
const defaultSrcChildDirectives = exports.defaultSrcChildDirectives = ['child-src', 'connect-src', 'font-src', 'frame-src', 'img-src', 'manifest-src', 'media-src', 'object-src', 'prefetch-src', 'script-src', 'script-src-elem', 'script-src-attr', 'style-src', 'style-src-elem', 'style-src-attr', 'worker-src'];
class CspDirectives {
  constructor() {
    (0, _defineProperty2.default)(this, "directives", new Map());
    (0, _defineProperty2.default)(this, "reportOnlyDirectives", new Map());
  }
  addDirectiveValue(directiveName, directiveValue, enforce = true) {
    const directivesMap = enforce ? this.directives : this.reportOnlyDirectives;
    let directive = directivesMap.get(directiveName);
    if (!directive) {
      directivesMap.set(directiveName, directive = new Set());
    }
    const normalizedDirectiveValue = normalizeDirectiveValue(directiveValue);
    // 'none' can not coexist with other values, and will be ignored by browsers.
    // In practice, this should only happen when a default rule defined above is set to 'none',
    // AND the administrator chose to specify a value via kibana.yml configuration. (e.g. see `object-src` above)
    if (directive.has(`'none'`) && normalizedDirectiveValue !== `'report-sample'`) {
      directive.delete(`'none'`);
    }
    directive.add(normalizedDirectiveValue);

    // If we are testing default-src 'none', then we need to add all expected child directives to the report-only policy
    // to prevent reports from being generated for those child directives.
    const enforcingDefaultSrcChildDirective = enforce && defaultSrcChildDirectives.includes(directiveName);
    if (this.isTestingDefaultSrc() && enforcingDefaultSrcChildDirective) {
      this.addDirectiveValue(directiveName, directiveValue, false);
    }
  }
  clearDirectiveValues(directiveName) {
    this.directives.delete(directiveName);
    this.reportOnlyDirectives.delete(directiveName);
  }
  getCspHeadersByDisposition() {
    return {
      enforceHeader: this.headerFromDirectives(this.directives),
      reportOnlyHeader: this.headerFromDirectives(this.reportOnlyDirectives)
    };
  }
  getCspHeader() {
    return this.headerFromDirectives(this.directives);
  }
  headerFromDirectives(directives) {
    return [...directives.entries()].map(([name, values]) => {
      return [name, ...values].join(' ');
    }).join('; ');
  }

  /**
   * Determines if we are currently testing the default-src 'none' configuration.
   * @returns True if we are testing default-src 'none', false otherwise.
   */
  isTestingDefaultSrc() {
    var _this$reportOnlyDirec;
    return this.reportOnlyDirectives.has('default-src') && (_this$reportOnlyDirec = this.reportOnlyDirectives.get('default-src')) !== null && _this$reportOnlyDirec !== void 0 && _this$reportOnlyDirec.has(`'none'`) ? true : false;
  }
  static fromConfig(firstConfig, ...otherConfigs) {
    const config = otherConfigs.reduce((acc, conf) => (0, _deepmerge.default)(acc, conf), firstConfig);
    const cspDirectives = new CspDirectives();

    // combining `default` report only directive configurations
    // it's important to add these before the enforced directives below so that we can handle report-only updates
    // in response to enforced directives (e.g., default-src 'none' testing)
    Object.entries(defaultReportOnlyRules).forEach(([key, values]) => {
      values === null || values === void 0 ? void 0 : values.forEach(value => {
        cspDirectives.addDirectiveValue(key, value, false);
      });
    });

    // combining `default` directive configurations
    Object.entries(defaultRules).forEach(([key, values]) => {
      values === null || values === void 0 ? void 0 : values.forEach(value => {
        cspDirectives.addDirectiveValue(key, value);
      });
    });

    // adding per-directive configuration
    const {
      enforceDirectives,
      reportOnlyDirectives
    } = parseConfigDirectives(config);
    for (const [directiveName, directiveValues] of enforceDirectives.entries()) {
      var _additionalRules$dire;
      const additionalValues = (_additionalRules$dire = additionalRules[directiveName]) !== null && _additionalRules$dire !== void 0 ? _additionalRules$dire : [];
      [...additionalValues, ...directiveValues].forEach(value => {
        cspDirectives.addDirectiveValue(directiveName, value);
      });
    }
    for (const [directiveName, directiveValues] of reportOnlyDirectives.entries()) {
      directiveValues.forEach(value => {
        cspDirectives.addDirectiveValue(directiveName, value, false);
      });
    }
    return cspDirectives;
  }
}
exports.CspDirectives = CspDirectives;
const parseConfigDirectives = cspConfig => {
  var _cspConfig$script_src, _cspConfig$worker_src, _cspConfig$style_src, _cspConfig$connect_sr, _cspConfig$default_sr, _cspConfig$font_src, _cspConfig$frame_src, _cspConfig$img_src, _cspConfig$object_src, _cspConfig$frame_ance, _cspConfig$report_uri, _cspConfig$report_to, _cspConfig$report_onl, _cspConfig$report_onl2, _cspConfig$report_onl4, _cspConfig$report_onl5;
  const enforceDirectives = new Map();
  const reportOnlyDirectives = new Map();
  if ((_cspConfig$script_src = cspConfig.script_src) !== null && _cspConfig$script_src !== void 0 && _cspConfig$script_src.length) {
    enforceDirectives.set('script-src', cspConfig.script_src);
  }
  if (cspConfig.disableUnsafeEval !== true) {
    var _enforceDirectives$ge;
    enforceDirectives.set('script-src', ["'unsafe-eval'", ...((_enforceDirectives$ge = enforceDirectives.get('script-src')) !== null && _enforceDirectives$ge !== void 0 ? _enforceDirectives$ge : [])]);
  }
  if ((_cspConfig$worker_src = cspConfig.worker_src) !== null && _cspConfig$worker_src !== void 0 && _cspConfig$worker_src.length) {
    enforceDirectives.set('worker-src', cspConfig.worker_src);
  }
  if ((_cspConfig$style_src = cspConfig.style_src) !== null && _cspConfig$style_src !== void 0 && _cspConfig$style_src.length) {
    enforceDirectives.set('style-src', cspConfig.style_src);
  }
  if ((_cspConfig$connect_sr = cspConfig.connect_src) !== null && _cspConfig$connect_sr !== void 0 && _cspConfig$connect_sr.length) {
    enforceDirectives.set('connect-src', cspConfig.connect_src);
  }
  if ((_cspConfig$default_sr = cspConfig.default_src) !== null && _cspConfig$default_sr !== void 0 && _cspConfig$default_sr.length) {
    enforceDirectives.set('default-src', cspConfig.default_src);
  }
  if ((_cspConfig$font_src = cspConfig.font_src) !== null && _cspConfig$font_src !== void 0 && _cspConfig$font_src.length) {
    enforceDirectives.set('font-src', cspConfig.font_src);
  }
  if ((_cspConfig$frame_src = cspConfig.frame_src) !== null && _cspConfig$frame_src !== void 0 && _cspConfig$frame_src.length) {
    enforceDirectives.set('frame-src', cspConfig.frame_src);
  }
  if ((_cspConfig$img_src = cspConfig.img_src) !== null && _cspConfig$img_src !== void 0 && _cspConfig$img_src.length) {
    enforceDirectives.set('img-src', cspConfig.img_src);
  }
  if ((_cspConfig$object_src = cspConfig.object_src) !== null && _cspConfig$object_src !== void 0 && _cspConfig$object_src.length) {
    enforceDirectives.set('object-src', cspConfig.object_src);
  }
  if ((_cspConfig$frame_ance = cspConfig.frame_ancestors) !== null && _cspConfig$frame_ance !== void 0 && _cspConfig$frame_ance.length) {
    enforceDirectives.set('frame-ancestors', cspConfig.frame_ancestors);
  }
  if ((_cspConfig$report_uri = cspConfig.report_uri) !== null && _cspConfig$report_uri !== void 0 && _cspConfig$report_uri.length) {
    enforceDirectives.set('report-uri', cspConfig.report_uri);
    reportOnlyDirectives.set('report-uri', cspConfig.report_uri);
  }
  if ((_cspConfig$report_to = cspConfig.report_to) !== null && _cspConfig$report_to !== void 0 && _cspConfig$report_to.length) {
    enforceDirectives.set('report-to', cspConfig.report_to);
    reportOnlyDirectives.set('report-to', cspConfig.report_to);
  }
  if ((_cspConfig$report_onl = cspConfig.report_only) !== null && _cspConfig$report_onl !== void 0 && (_cspConfig$report_onl2 = _cspConfig$report_onl.form_action) !== null && _cspConfig$report_onl2 !== void 0 && _cspConfig$report_onl2.length) {
    var _cspConfig$report_onl3;
    reportOnlyDirectives.set('form-action', (_cspConfig$report_onl3 = cspConfig.report_only) === null || _cspConfig$report_onl3 === void 0 ? void 0 : _cspConfig$report_onl3.form_action);
  }
  if ((_cspConfig$report_onl4 = cspConfig.report_only) !== null && _cspConfig$report_onl4 !== void 0 && (_cspConfig$report_onl5 = _cspConfig$report_onl4.object_src) !== null && _cspConfig$report_onl5 !== void 0 && _cspConfig$report_onl5.length) {
    var _cspConfig$report_onl6;
    reportOnlyDirectives.set('object-src', (_cspConfig$report_onl6 = cspConfig.report_only) === null || _cspConfig$report_onl6 === void 0 ? void 0 : _cspConfig$report_onl6.object_src);
  }
  return {
    enforceDirectives,
    reportOnlyDirectives
  };
};
const keywordTokens = ['none', 'self', 'strict-dynamic', 'report-sample', 'unsafe-inline', 'unsafe-eval', 'unsafe-hashes', 'unsafe-allow-redirects'];
function normalizeDirectiveValue(value) {
  if (keywordTokens.includes(value)) {
    return `'${value}'`;
  }
  return value;
}