#ifndef RM_CONFIG_H
#define RM_CONFIG_H

#define INSTALL_PREFIX "/usr"

#define HAVE_BLKID         (1)
#define HAVE_LIBINTL       (1)
#define HAVE_LIBELF        (1)
#define HAVE_JSON_GLIB     (1)
#define HAVE_GIO_UNIX      (1)
#define HAVE_FIEMAP        (1)
#define HAVE_XATTR         (1)
#define HAVE_LXATTR        (1)
#define HAVE_SHA512        (1)
#define HAVE_BIGFILES      (1)
#define HAVE_STAT64        (1)
#define HAVE_BIG_OFF_T     (1)
#define HAVE_SYSBLOCK      (1)
#define HAVE_LINUX_LIMITS  (1)
#define HAVE_POSIX_FADVISE (1)
#define HAVE_BTRFS_H       (1)
#define HAVE_LINUX_FS_H    (1)
#define HAVE_UNAME         (1)
#define HAVE_SYSMACROS_H   (1)
#define HAVE_MM_CRC32_U64  (1)
#define HAVE_BUILTIN_CPU_SUPPORTS (1)

/* define here so rmlint and hash utility can both access */
#define RM_DEFAULT_DIGEST RM_DIGEST_BLAKE2B

#define RM_VERSION "2.10.3"
#define RM_VERSION_MAJOR 2
#define RM_VERSION_MINOR 10
#define RM_VERSION_PATCH 3
#define RM_VERSION_NAME  "Ludicrous Lemur"
#define RM_VERSION_GIT_REVISION "3a1c0a2"

#define RM_MANPAGE_USE_PAGER (1)

/* Might come in useful */
#define RM_CHECK_VERSION(X,Y,Z) (0  \
    || X <= RM_VERSION_MAJOR        \
    || Y <= RM_VERSION_MINOR        \
    || Z <= RM_VERSION_MICRO        \
)

/* These colors should only be used with the rm_log_* macros below */
#define RED    "\x1b[31;01m"
#define YELLOW "\x1b[33;01m"
#define RESET  "\x1b[0m"
#define GREEN  "\x1b[32;01m"
#define BLUE   "\x1b[34;01m"

#include <errno.h>

#if HAVE_LINUX_LIMITS
#include <linux/limits.h>
#else
# ifndef PATH_MAX
#  define PATH_MAX 4096
# endif
#endif

#define rm_log_debug(...) \
    g_log("rmlint", G_LOG_LEVEL_DEBUG, __VA_ARGS__)
#define rm_log_info(...) \
    g_log("rmlint", G_LOG_LEVEL_INFO, __VA_ARGS__)
#define rm_log_warning(...) \
    g_log("rmlint", G_LOG_LEVEL_WARNING, __VA_ARGS__)
#define rm_log_error(...) \
    g_log("rmlint", G_LOG_LEVEL_CRITICAL, __VA_ARGS__)

#define rm_log_perror(message)                                                           \
    if(errno) {                                                                         \
        rm_log_error_line("%s:%d: %s: %s", __FILE__, __LINE__, message, g_strerror(errno)); \
    }                                                                                   \

#define rm_log_perrorf(message, ...)                                \
    if(errno) {                                                    \
        int _errsv = errno;                                         \
        char *msg = g_strdup_printf(message, __VA_ARGS__);          \
        rm_log_error_line("%s:%d: %s: %s", __FILE__, __LINE__, msg, \
                          g_strerror(_errsv));                      \
        g_free(msg);                                                \
    }

#define _UNUSED G_GNUC_UNUSED
#define LLU G_GUINT64_FORMAT
#define LLI G_GINT64_FORMAT


#include <stdint.h> /* for UINTPTR_MAX */
#define RM_PLATFORM_32 (UINTPTR_MAX == 0xffffffff)
#define RM_PLATFORM_64 (UINTPTR_MAX == 0xffffffffffffffff)

#if defined(__APPLE__) && defined(__MACH__)
# define RM_IS_APPLE 1
#else
# define RM_IS_APPLE 0
#endif

#ifdef __CYGWIN__
# define RM_IS_CYGWIN 1
#else
# define RM_IS_CYGWIN 0
#endif

#include <glib.h>

/* Printf formatting for large integers */
#include <inttypes.h>

#define RM_GETTEXT_PACKAGE "rmlint"

#if HAVE_LIBINTL
#  define _(String) gettext (String)
#  define gettext_noop(String) String
#  define N_(String) gettext_noop (String)
#  include <libintl.h>
#else
#  define _(String) (String)
#  define gettext_noop(String) String
#  define N_(String) gettext_noop (String)
#endif

static inline GMutex* rm_log_get_mutex(void) {
    static GMutex RM_LOG_MTX;
    return &RM_LOG_MTX;
}

#define RM_LOG_INIT \
    g_mutex_init(rm_log_get_mutex());

typedef guint64 RmOff;

/* Stupid macros to make printing error lines easier */
#define rm_log_error_prefix()    \
    rm_log_error(RED);           \
    rm_log_error(_("ERROR"));    \
    rm_log_error(": "RESET);     \

#define rm_log_warning_prefix()   \
    rm_log_warning(YELLOW);       \
    rm_log_warning(_("WARNING")); \
    rm_log_warning(": "RESET);    \

#define rm_log_info_prefix()      \
    rm_log_warning(GREEN);        \
    rm_log_warning(_("INFO"));    \
    rm_log_warning(": "RESET);    \

#define rm_log_debug_prefix()    \
    rm_log_debug(BLUE);          \
    rm_log_debug(_("DEBUG"));    \
    rm_log_debug(": "RESET);     \

///////////////

#define rm_log_error_line(...)           \
    g_mutex_lock(rm_log_get_mutex());    \
    rm_log_error_prefix()                \
    rm_log_error(__VA_ARGS__);           \
    rm_log_error("\n");                  \
    g_mutex_unlock(rm_log_get_mutex());  \

#define rm_log_warning_line(...)         \
    g_mutex_lock(rm_log_get_mutex());    \
    rm_log_warning_prefix()              \
    rm_log_warning(__VA_ARGS__);         \
    rm_log_warning("\n");                \
    g_mutex_unlock(rm_log_get_mutex());  \

#define rm_log_info_line(...)            \
    g_mutex_lock(rm_log_get_mutex());    \
    rm_log_info_prefix()                 \
    rm_log_warning(__VA_ARGS__);         \
    rm_log_warning("\n");                \
    g_mutex_unlock(rm_log_get_mutex());  \

#define rm_log_debug_line(...)           \
    g_mutex_lock(rm_log_get_mutex());    \
    rm_log_debug_prefix()                \
    rm_log_debug(__VA_ARGS__);           \
    rm_log_debug("\n");                  \
    g_mutex_unlock(rm_log_get_mutex());  \

/* Domain for reporting errors. Needed by GOptions */
#define RM_ERROR_QUARK (g_quark_from_static_string("rmlint"))

#ifdef RM_DEBUG
    #undef NDEBUG
#else
    #ifndef NDEBUG
        #define NDEBUG
    #endif
    #ifdef __GNUC__
        #define INLINE inline __attribute__((__always_inline__))
    #elif defined(__CLANG__)
        #if __has_attribute(__always_inline__)
            #define INLINE inline __attribute__((__always_inline__))
        #endif
    #elif defined(_MSC_VER)
        #define INLINE __forceinline
    #endif
#endif

#ifndef INLINE
    #define INLINE inline
#endif

#ifdef __GNUC__
    #define WARN_UNUSED_RESULT __attribute__((warn_unused_result))
#elif defined(__CLANG__)
    #define WARN_UNUSED_RESULT __attribute__((warn_unused_result))
#else
    /* give up */
    #define WARN_UNUSED_RESULT
#endif

#define _RM_OFFSET_DEBUG 0

#endif
