/*
 *  This file is part of rmlint.
 *
 *  rmlint is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  rmlint is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with rmlint.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authors:
 *
 *  - Christopher <sahib> Pahl 2010-2017 (https://github.com/sahib)
 *  - Daniel <SeeSpotRun> T.   2014-2017 (https://github.com/SeeSpotRun)
 *
 * Hosted on http://github.com/sahib/rmlint
 *
 */

#include "../formats.h"
#include "../utilities.h"
#include "../preprocess.h"

#include <glib.h>
#include <stdio.h>
#include <string.h>

static const char PY_SOURCE[] = "#!/usr/bin/env python3\n"
"# encoding: utf-8\n"
"\n"
"\"\"\" This file is part of rmlint.\n"
"\n"
"rmlint is free software: you can redistribute it and/or modify\n"
"it under the terms of the GNU General Public License as published by\n"
"the Free Software Foundation, either version 3 of the License, or\n"
"(at your option) any later version.\n"
"\n"
"rmlint is distributed in the hope that it will be useful,\n"
"but WITHOUT ANY WARRANTY; without even the implied warranty of\n"
"MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the\n"
"GNU General Public License for more details.\n"
"\n"
"You should have received a copy of the GNU General Public License\n"
"along with rmlint.  If not, see <http://www.gnu.org/licenses/>.\n"
"\n"
"Authors:\n"
"\n"
"- Christopher <sahib> Pahl 2010-2017 (https://github.com/sahib)\n"
"- Daniel <SeeSpotRun> T.   2014-2017 (https://github.com/SeeSpotRun)\n"
"\"\"\"\n"
"\n"
"# This is the python remover utility shipped inside the rmlint binary.\n"
"# The 200 lines source presented below is meant to be clean and hackable.\n"
"# It is intended to be used for corner cases where the built-in sh formatter\n"
"# is not enough or as an alternative to it. By default it works the same.\n"
"#\n"
"# Disable a few pylint warnings, in case someone integrates into scripts:\n"
"# pylint: disable=unused-argument,missing-docstring,invalid-name\n"
"# pylint: disable=redefined-outer-name,unused-variable\n"
"\n"
"# Python2 compat:\n"
"from __future__ import print_function\n"
"\n"
"import os\n"
"import sys\n"
"import pwd\n"
"import json\n"
"import shutil\n"
"import filecmp\n"
"import argparse\n"
"import subprocess\n"
"\n"
"CURRENT_UID = os.geteuid()\n"
"CURRENT_GID = pwd.getpwuid(CURRENT_UID).pw_gid\n"
"\n"
"USE_COLOR = sys.stdout.isatty() and sys.stderr.isatty()\n"
"COLORS = {\n"
"    'red':    \"\x1b[0;31m\" if USE_COLOR else \"\",\n"
"    'blue':   \"\x1b[1;34m\" if USE_COLOR else \"\",\n"
"    'green':  \"\x1b[0;32m\" if USE_COLOR else \"\",\n"
"    'yellow': \"\x1b[0;33m\" if USE_COLOR else \"\",\n"
"    'reset':  \"\x1b[0m\" if USE_COLOR else \"\",\n"
"}\n"
"\n"
"\n"
"def original_check(path, original, be_paranoid=True):\n"
"    try:\n"
"        stat_p, stat_o = os.stat(path), os.stat(original)\n"
"        if (stat_p.st_dev, stat_p.st_ino) == (stat_o.st_dev, stat_o.st_ino):\n"
"            print('{c[red]}Same inode; ignoring:{c[reset]} {o} <=> {p}'.format(\n"
"                c=COLORS, o=original, p=path))\n"
"            return False\n"
"\n"
"        if stat_p.st_size != stat_o.st_size:\n"
"            print('{c[red]}Size differs; ignoring:{c[reset]} '\n"
"                  '{o} <=> {p}'.format(c=COLORS, o=original, p=path))\n"
"            return False\n"
"\n"
"        if be_paranoid and not filecmp.cmp(path, original):\n"
"            print('{c[red]}Content differs; ignoring:{c[reset]} '\n"
"                  '{o} <=> {p}'.format(c=COLORS, o=original, p=path))\n"
"            return False\n"
"\n"
"        return True\n"
"    except OSError as exc:\n"
"        print('{c[red]}{exc}{c[reset]}'.format(c=COLORS, exc=exc))\n"
"        return False\n"
"\n"
"\n"
"def handle_duplicate_dir(path, original, **kwargs):\n"
"    if not args.dry_run:\n"
"        shutil.rmtree(path)\n"
"\n"
"\n"
"def handle_duplicate_file(path, original, args, **kwargs):\n"
"    if original_check(path, original['path'], be_paranoid=args.paranoid):\n"
"        if not args.dry_run:\n"
"            os.remove(path)\n"
"\n"
"\n"
"def handle_unique_file(path, **kwargs):\n"
"    pass  # doesn't need any handling.\n"
"\n"
"\n"
"def handle_empty_dir(path, **kwargs):\n"
"    if not args.dry_run:\n"
"        os.rmdir(path)\n"
"\n"
"\n"
"def handle_empty_file(path, **kwargs):\n"
"    if not args.dry_run:\n"
"        os.remove(path)\n"
"\n"
"\n"
"def handle_nonstripped(path, **kwargs):\n"
"    if not args.dry_run:\n"
"        subprocess.call([\"strip\", \"--strip-debug\", path])\n"
"\n"
"\n"
"def handle_badlink(path, **kwargs):\n"
"    if not args.dry_run:\n"
"        os.remove(path)\n"
"\n"
"\n"
"def handle_baduid(path, **kwargs):\n"
"    if not args.dry_run:\n"
"        os.chown(path, args.user, -1)\n"
"\n"
"\n"
"def handle_badgid(path, **kwargs):\n"
"    if not args.dry_run:\n"
"        os.chown(path, -1, args.group)\n"
"\n"
"\n"
"def handle_badugid(path, **kwargs):\n"
"    if not args.dry_run:\n"
"        os.chown(path, args.user, args.group)\n"
"\n"
"\n"
"OPERATIONS = {\n"
"    \"duplicate_dir\": handle_duplicate_dir,\n"
"    \"duplicate_file\": handle_duplicate_file,\n"
"    \"unique_file\": handle_unique_file,\n"
"    \"emptydir\": handle_empty_dir,\n"
"    \"emptyfile\": handle_empty_file,\n"
"    \"nonstripped\": handle_nonstripped,\n"
"    \"badlink\": handle_badlink,\n"
"    \"baduid\": handle_baduid,\n"
"    \"badgid\": handle_badgid,\n"
"    \"badugid\": handle_badugid,\n"
"}\n"
"\n"
"\n"
"def exec_operation(item, original=None, args=None):\n"
"    try:\n"
"        OPERATIONS[item['type']](\n"
"            item['path'], original=original, item=item, args=args)\n"
"    except OSError as err:\n"
"        print('{c[red]}# {err}{c[reset]}'.format(\n"
"            err=err, c=COLORS\n"
"        ), file=sys.stderr)\n"
"\n"
"\n"
"MESSAGES = {\n"
"    'duplicate_dir':    '{c[yellow]}Deleting duplicate directory:',\n"
"    'duplicate_file':   '{c[yellow]}Deleting duplicate:',\n"
"    'unique_file':      'checking',\n"
"    'emptydir':         '{c[green]}Deleting empty directory:',\n"
"    'emptyfile':        '{c[green]}Deleting empty file:',\n"
"    'nonstripped':      '{c[green]}Stripping debug symbols:',\n"
"    'badlink':          '{c[green]}Deleting bad symlink:',\n"
"    'baduid':           '{c[green]}chown {u}',\n"
"    'badgid':           '{c[green]}chgrp {g}',\n"
"    'badugid':          '{c[green]}chown {u}:{g}',\n"
"}\n"
"\n"
"ORIGINAL_MESSAGES = {\n"
"    'duplicate_file':   '{c[green]}Keeping original:  ',\n"
"    'duplicate_dir':    '{c[green]}Keeping original directory:  ',\n"
"}\n"
"\n"
"\n"
"def main(args, data):\n"
"    last_original_item = None\n"
"\n"
"    # Process header and footer, if present\n"
"    header, footer = [], []\n"
"    if data[0].get('description'):\n"
"        header = data.pop(0)\n"
"    if data[-1].get('total_files'):\n"
"        footer = data.pop(-1)\n"
"\n"
"    if not args.no_ask and not args.dry_run:\n"
"        print('rmlint was executed in the following way:\\n',\n"
"              header.get('args'),\n"
"              '\\n\\nPress Enter to continue and perform modifications, '\n"
"              'or CTRL-C to exit.'\n"
"              '\\nExecute this script with -d to disable this message.',\n"
"              file=sys.stderr)\n"
"        sys.stdin.read(1)\n"
"\n"
"    for item in data:\n"
"        progress_prefix = '{c[blue]}[{p:3}%]{c[reset]} '.format(\n"
"            c=COLORS, p=item['progress'])\n"
"\n"
"        if item['is_original']:\n"
"            msg = ORIGINAL_MESSAGES[item['type']].format(c=COLORS)\n"
"            print('{prog}{v}{c[reset]} {path}'.format(\n"
"                c=COLORS, prog=progress_prefix, v=msg, path=item['path']))\n"
"            last_original_item = item\n"
"            # Do not handle originals.\n"
"            continue\n"
"\n"
"        msg = MESSAGES[item['type']].format(\n"
"            c=COLORS, u=args.user, g=args.group)\n"
"        print('{prog}{v}{c[reset]} {path}'.format(\n"
"            c=COLORS, prog=progress_prefix, v=msg, path=item['path']))\n"
"        exec_operation(item, original=last_original_item, args=args)\n"
"\n"
"    print('{c[blue]}[100%] Done!{c[reset]}'.format(c=COLORS))\n"
"\n"
"\n"
"if __name__ == '__main__':\n"
"    parser = argparse.ArgumentParser(\n"
"        description='Handle the files in a JSON output of rmlint.'\n"
"    )\n"
"\n"
"    parser.add_argument(\n"
"        'json_files', metavar='json_file', nargs='*', default=['.rmlint.json'],\n"
"        help='A JSON output of rmlint to handle (can be given multiple times)'\n"
"    )\n"
"    parser.add_argument(\n"
"        '-n', '--dry-run', action='store_true',\n"
"        help='Do not perform any modifications, just print what would be '\n"
"             'done. (implies -d)'\n"
"    )\n"
"    parser.add_argument(\n"
"        '-d', '--no-ask', action='store_true', default=False,\n"
"        help='Do not ask for confirmation before running.'\n"
"    )\n"
"    parser.add_argument(\n"
"        '-p', '--paranoid', action='store_true', default=False,\n"
"        help='Recheck that files are still identical before removing '\n"
"             'duplicates.'\n"
"    )\n"
"    parser.add_argument(\n"
"        '-u', '--user', type=int, default=CURRENT_UID,\n"
"        help='Numerical uid for chown operations'\n"
"    )\n"
"    parser.add_argument(\n"
"        '-g', '--group', type=int, default=CURRENT_GID,\n"
"        help='Numerical gid for chgrp operations'\n"
"    )\n"
"\n"
"    args = parser.parse_args()\n"
"    json_docs = []\n"
"    for json_file in args.json_files:\n"
"        try:\n"
"            with open(json_file) as f:\n"
"                j = json.load(f)\n"
"            json_docs.append(j)\n"
"        except IOError as err:      # Cannot open file\n"
"            print(err, file=sys.stderr)\n"
"            sys.exit(-1)\n"
"        except ValueError as err:   # File is not valid JSON\n"
"            print('{}: {}'.format(err, json_file), file=sys.stderr)\n"
"            sys.exit(-1)\n"
"\n"
"    try:\n"
"        if args.dry_run:\n"
"            print(\n"
"                '{c[green]}#{c[reset]} '\n"
"                'This is a dry run. Nothing will be modified.'.format(\n"
"                    c=COLORS\n"
"                )\n"
"            )\n"
"\n"
"        for json_doc in json_docs:\n"
"            main(args, json_doc)\n"
"\n"
"        if args.dry_run:\n"
"            print(\n"
"                '{c[green]}#{c[reset]} '\n"
"                'This was a dry run. Nothing was modified.'.format(\n"
"                    c=COLORS\n"
"                )\n"
"            )\n"
"    except KeyboardInterrupt:\n"
"        print('\\ncanceled.')\n"
"";

typedef struct RmFmtHandlerPy {
    /* must be first */
    RmFmtHandler parent;

    /* HACK: Actual handler might be bigger, add padding. */
    char _dummy[1024];

    /* Filehandle to pass to the JSON Formatter. */
    FILE *json_out;

} RmFmtHandlerPy;

/////////////////////////
//  ACTUAL CALLBACKS   //
/////////////////////////

static void rm_fmt_head(RmSession *session, RmFmtHandler *parent, FILE *out) {
    RmFmtHandlerPy *self = (RmFmtHandlerPy *)parent;

    if(fwrite(PY_SOURCE, 1, sizeof(PY_SOURCE) - 1, out) <= 0) {
        rm_log_perror("Failed to write python script");
        return;
    }

    if(fchmod(fileno(out), S_IRUSR | S_IWUSR | S_IXUSR) == -1) {
        rm_log_perror("Could not chmod +x python-script");
    }

    self->json_out = fopen(".rmlint.json", "w");
    if(self->json_out == NULL)  {
        return;
    }

    /* Delegate */
    extern RmFmtHandler *JSON_HANDLER;
    JSON_HANDLER->head(session, (RmFmtHandler *)self, self->json_out);
}

static void rm_fmt_foot(RmSession *session, RmFmtHandler *parent, _UNUSED FILE *out) {
    RmFmtHandlerPy *self = (RmFmtHandlerPy *)parent;
    if(self->json_out == NULL) {
        return;
    }

    /* Delegate */
    extern RmFmtHandler *JSON_HANDLER;
    JSON_HANDLER->foot(session, (RmFmtHandler *)self, self->json_out);

    fclose(self->json_out);
}

static void rm_fmt_elem(
    RmSession *session,
    RmFmtHandler *parent,
    _UNUSED FILE *out, RmFile *file
) {
    RmFmtHandlerPy *self = (RmFmtHandlerPy *)parent;
    if(self->json_out == NULL) {
        return;
    }

    /* Delegate */
    extern RmFmtHandler *JSON_HANDLER;
    JSON_HANDLER->elem(session, (RmFmtHandler *)self, self->json_out, file);
}

static RmFmtHandlerPy PY_HANDLER_IMPL = {
    /* Initialize parent */
    .parent = {
        .size = sizeof(PY_HANDLER_IMPL),
        .name = "py",
        .head = rm_fmt_head,
        .elem = rm_fmt_elem,
        .prog = NULL,
        .foot = rm_fmt_foot,
        .valid_keys = {NULL},
    }
};

RmFmtHandler *PY_HANDLER = (RmFmtHandler *) &PY_HANDLER_IMPL;
