"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.messageDefaultResponse = exports.isMessage = exports.knownPacketID = exports.staticEvents = exports.RPCActivity = void 0;
/**
 * The type of the activity displayed as a status message with its name (e.g.
 * *Playing \<game\>* or *Listening to \<music\>*).
 */
var RPCActivity;
(function (RPCActivity) {
    RPCActivity[RPCActivity["Game"] = 0] = "Game";
    RPCActivity[RPCActivity["Listening"] = 1] = "Listening";
    RPCActivity[RPCActivity["Streaming"] = 2] = "Streaming";
    RPCActivity[RPCActivity["Watching"] = 3] = "Watching";
    RPCActivity[RPCActivity["Custom"] = 4] = "Custom";
    RPCActivity[RPCActivity["Competing"] = 5] = "Competing";
})(RPCActivity || (exports.RPCActivity = RPCActivity = {}));
/** Static, hard-coded event objects sent by transports at certain conditions.*/
exports.staticEvents = Object.freeze({
    /**
     * A fake, hard-coded Discord event to spoof the presence of
     * official Discord client (which makes browser to actually start a
     * communication with the DisConnection).
     */
    ready: {
        cmd: "DISPATCH",
        data: {
            v: 1,
            config: null,
            user: null
        },
        evt: "READY",
        nonce: null
    }
});
/**
 * Known *identifiers* of packets, which tends to be either a `code` of
 * `cmd`, `events` (`evt`) or `type` (for `DEEP_LINK`).
 */
exports.knownPacketID = Object.freeze({
    codes: Object.freeze(["INVITE_BROWSER", "GUILD_TEMPLATE_BROWSER", "AUTHORIZE", "DEEP_LINK", "SET_ACTIVITY"]),
    types: Object.freeze(["CHANNEL"]),
    events: Object.freeze(["READY", "ERROR"])
});
/**
 * Generic response checker, assumes Discord will do requests of certain type
 * based on `cmd` and `argsType` values.
 */
function isMessage(data, cmd, argsType) {
    /** Verifies if given value satisfies record of the given type. */
    function checkRecord(record, keys, arg) {
        for (const key of keys)
            if (typeof record[key] !== arg)
                return false;
        return true;
    }
    if (typeof data !== "object" || data === null || data.constructor !== Object)
        return false;
    // Check first if it is any kind of Discord message.
    if (!checkRecord(data, ["cmd", "nonce"], "string"))
        return false;
    if (typeof data.args !== "object")
        return false;
    // Check "cmd" value
    if (typeof cmd === "string") {
        if (data.cmd !== cmd)
            return false;
    }
    else if (Array.isArray(cmd)) {
        if (!cmd.includes(data.cmd))
            return false;
    }
    // Check "args.type" value
    if (argsType !== undefined && typeof data.args.params === "object")
        switch (argsType) {
            case "CHANNEL":
                if (!(checkRecord(data.args.params, ["guildId", "channelId", "search", "fingerprint"], "string") || data.args.params.channelId === undefined))
                    return false;
        }
    // All is "good enough"
    return true;
}
exports.isMessage = isMessage;
function messageDefaultResponse(message) {
    const browserReq = message.cmd.endsWith("_BROWSER");
    return Object.freeze({
        cmd: message.cmd,
        data: browserReq ? {
            code: message.args["code"] ?? null,
            ...(message.cmd === "GUILD_TEMPLATE_BROWSER" ? {
                guildTemplate: {
                    code: message.args["code"] ?? null
                }
            } : {})
        } : null,
        ...(browserReq ? {} : { evt: null }),
        nonce: message.nonce
    });
}
exports.messageDefaultResponse = messageDefaultResponse;
//# sourceMappingURL=packet.js.map