import { type colors } from "@spacingbat3/kolor";
import { Message, code, type, messageDefaultResponse } from "./packet";
/**
 * Flattened combination of `codes` and `types` from {@link knownMsgEl} used as
 * an array of hook names.
 */
declare const hookNames: readonly ("INVITE_BROWSER" | "GUILD_TEMPLATE_BROWSER" | "AUTHORIZE" | "SET_ACTIVITY" | "DEEP_LINK_CHANNEL")[];
/** Alias type for single hook name. */
type hookName = typeof hookNames[number];
/** An object that maps given hook names to their respective function argument tuple. */
export type HookSignatures = {
    [P in hookName]: P extends `${infer C extends "DEEP_LINK"}_${infer T extends type}` ? [request: Message<C, T>, origin: string | null] : P extends infer C extends code ? [request: Message<C, never>, origin: string | null] : never;
};
/** An alias which generates a function type from its signatures based on hook name. */
export type HookFn<T extends hookName> = (...args: HookSignatures[T]) => Promise<undefined | number>;
/**
 * An alias to list of the colors allowed to be used with `@spacingbat3/kolor`
 * library. Used for tag color in the log message.
 */
export type fgColor = Exclude<keyof typeof colors, `bg${Capitalize<keyof typeof colors>}`>;
/**
 * A specification which defines Discord communication protocol used within
 * DisConnection. It is used for implementing various *transports*, like
 * WebSocket server or UNIX socket (IPC). This class is not designed to be used
 * directly, but is meant to be extended by given transport implementation.
 */
export declare abstract class Protocol<T extends string = string> {
    #private;
    /**
     * A name which indicates the given implementation of the protocol.
     */
    abstract readonly name: T;
    /**
     * A {@link name} variant which contains only English lowercase letters with
     * other incompatible characters replaced with underscore (`_`).
     */
    get safeName(): import("@spacingbat3/lss").sanitizeResult<T, "a-z", "_", "both">;
    /** A way to stop the server while {@link destroy}-ing the class structure. */
    protected abstract stopServer(): void;
    /** Writes a regular message (`log`) in the console to `stdout`. */
    log(...args: unknown[]): void;
    /** Writes an error message to `stderr`. */
    protected error(...args: unknown[]): void;
    /**
     * Writes a debug message, which will be visible whenever `NODE_DEBUG`
     * includes a `disconnection-{{@link #safeName}}`.
     */
    protected debug(...args: unknown[]): void;
    /**
     * Whenever this class has been *destroyed*.
     *
     * @see {@link destroy} for more details about this state.
     */
    isDestroyed(): boolean;
    /**
     * Removes references in class properties and methods either by replacing
     * values with a reference to the function that throws an error (for methods
     * and some required properties using getters) or sets them to a nullish value
     * (for optional properties).
     *
     * As it is hard to guarantee the Garbage Collector will ever deallocate
     * memory after dereferencing all of the objects, the goal is to prevent API
     * consumers from using class method that no longer make sense than implement
     * any kind of the memory cleanup logic.
     *
     * **Note: This operation is designed to be irreversible!** You will have to
     * initialize the new class instance if you want to use given transport again.
     *
     * @throws {@link Error} in case object has been already destroyed.
     *
     * @since v1.1.0
     */
    destroy(): void;
    /**
     * Adds a hook to the given hook list if it doesn't exist in it.
     *
     * @param {T extends HookName} name - A name of hook list.
     * @param value - A function that will be added to hook list.
     *
     * @returns number of hooks of given key or `false` if value were added before
     * @throws {@link TypeError} on invalid function parameter types.
     * @since v1.0.0
     */
    addHook<T extends hookName>(name: T, value: HookFn<T>): number | false;
    /**
     * Removes given hook function from give the hook list.
     *
     * @param name - A name of hook list.
     *
     * @returns whenever hook has been deleted
     * @throws {@link TypeError} on invalid function parameter types.
     * @since v1.0.0
     */
    removeHook<T extends hookName>(name: T, value: HookFn<T>): boolean;
    /**
     * Removes **all** hooks from the given hook list.
     *
     * @param name - A name of hook list.
     *
     * @returns if hook list wasn't empty before removing — values from it
     * @throws {@link TypeError} on invalid hook list name.
     * @since v1.0.0
     */
    removeAllHooks<T extends hookName>(name: T): boolean;
    /**
     * Lists all hooks from the given hook list.
     *
     * @param name - A name of hook list.
     *
     * @returns `Array` of hooks
     * @throws {@link TypeError} on invalid hook list name.
     * @since v1.0.0
     */
    getHooks<T extends hookName>(name: T): HookFn<T>[];
    /**
     * Whenever any of hooks will execute by server.
     *
     * @param name - A name of hook list.
     *
     * @returns whenever hooks are *active*
     * @throws {@link TypeError} on invalid hook list name.
     * @since v1.0.0
     */
    anyHooksActive<T extends hookName>(name: T): boolean;
    /**
     * Switches state of a given hook list, which can either disable it or not.
     *
     * @param name - A name of hook list.
     * @param active - New state of hooks. Defaults to negation of previous state.
     *
     * @returns current state of given hook (i.e if it is active or not)
     * @throws {@link TypeError} on invalid function parameter types.
     * @since v1.0.0
     */
    toggleHooks<T extends hookName>(name: T, active?: boolean): boolean;
    constructor(cConsole?: Console | null, color?: fgColor);
    /**
     * This function maps incoming messages from transports to outgoing messages
     * with partially-filled data, i.e. nothing is being resolved as it takes
     * place in the official Discord client.
     *
     * This is used in place of default responses.
     *
     * @param message - Incoming message from transports.
     *
     * @returns Outgoing message that can be send as a response.
     *
     * @since v1.0.0
     *
     * @deprecated This was never expected to be a part of public API.
     */
    static readonly messageResponse: typeof messageDefaultResponse;
}
export {};
