import { Server } from "ws";
import { Protocol, fgColor } from "../common/protocol";
/**
 * A list of standard status codes used within WebSocket communication at
 * connection close. Currently, not all are documented there, although all were
 * listed, with some additional ones took from MDN.
 *
 * Reference: [MDN], [RFC 6455].
 *
 * [MDN]: https://developer.mozilla.org/en-US/docs/Web/API/CloseEvent/code "CloseEvent.close – Web APIs | MDN"
 * [RFC 6455]: https://www.rfc-editor.org/rfc/rfc6455.html#section-7.4.1  "RFC 6455: The WebSocket protocol."
 */
export declare const enum WebSocketClose {
    /**
     * Indicates that connection went successfully.
     */
    Ok = 1000,
    /**
     * Emitted when endpoint is going away, e.g. on navigation or server failure.
     */
    GoingAway = 1001,
    /**
     * Emitted once a protocol error occurs.
     */
    ProtocolError = 1002,
    /**
     * Emitted once server received a kind of data it couldn't accept, e.g. binary
     * message on endpoints supporting only UTF-8 encoded data.
     */
    UnsupportedData = 1003,
    /**
     * **Reserved**. It currently has no meaning, but that might change in the
     * future.
     */
    Reserved = 1004,
    /**
     * **Reserved**. Indicates lack of the status/code, although it was expected.
     */
    NoStatusReceived = 1005,
    /**
     * **Reserved**. Emitted when connection was closed abnormally, where status
     * code was expected.
     */
    AbnormalClosure = 1006,
    /**
     * Indicates that server received a message with inconsistent data structure,
     * e.g. a mixed UTF-8 encoded message that also includes the unrecognizable
     * binary data.
     */
    InvalidPayload = 1007,
    /**
     * Indicates that sent message violates the server's policy. This code is
     * meant to be a generic status code that can be used if there's no suitable
     * more suitable status codes (like {@link UnsupportedData} or {@link MessageTooBig}).
     * It could also be used if details about the policy should be hidden.
     */
    PolicyViolation = 1008,
    /**
     * Indicates that received payload by the server is too large to be processed.
     */
    MessageTooBig = 1009,
    MandatoryExtension = 1010,
    InternalError = 1011,
    ServiceRestart = 1012,
    /**
     * Emitted when server is terminating connection due to the temporarily
     * condition, e.g. server overload.
     */
    TryAgainLater = 1013,
    BadGateway = 1014
}
/**
 * An information about the WebSocket server, like the reserved number of the
 * port and reference to the {@link Server} class.
 */
export interface ServerDetails {
    server: Server;
    port: number;
}
/**
 * Implements Discord client communication {@link Protocol} (between Discord
 * browser or any software with Discord integrations) via WebSocket server.
 */
export declare class WebSocketProtocol extends Protocol<"WebSocket"> {
    name: "WebSocket";
    stopServer(): void;
    details?: Promise<ServerDetails>;
    /** Creates new instance of {@link WebSocketProtocol} class.
     *
     * @param validOrigins - Whitelist of client origins as {@link RegExp} patterns or strings.
     * @param console - [`Console`](https://nodejs.org/api/console.html#console) instance used within class (defaults to {@link global.console}).
     *
     * @throws - {@link Error} if server couldn't be created within a given port range.
     */
    constructor(validOrigins: (RegExp | string)[], console?: Console | null, color?: fgColor);
}
